/*
    Simple Graphics Framework (SGF) - header file

    Copyright 2023 Arnold Beiland

    Simple Graphics Framework is free software: you can redistribute it and/or
    modify it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along with
    this program (look for a file named COPYING in the top directory). If not, see
    <https://www.gnu.org/licenses/>.
*/

#ifndef _SGF_H
#define _SGF_H

#include <fstream>
#include <cstdint>
using std::uint8_t;

// represents the color of a pixel,
// color components are one-byte integers (0..255)
struct Color {
    uint8_t r, g, b;
};

// can be used for logging status info
// (instead of cout)
extern std::ofstream logfile;

// drawing functions provided by the framework:

void draw_pixel(int row, int col, Color color);

void draw_text(int row, int col,
               Color color, int size,
               const char *text);


// The following functions have to be implemented in user code:

// called once when the program starts
void initialize();

// called every time the window needs to be rendered,
// should render all graphical content via calls to
// draw_pixel and draw_text
void render(int width, int height);

// mouse events:
bool on_scroll(int delta); // delta can be positive or negative
bool on_move(int row, int col); // row and col are the current position
bool on_mouse_down();
bool on_mouse_up();

// key events, the key parameter can be any
// key code (including the constants below);
// regular characters have the same code as
// in ASCII
bool on_key_down(int key);
bool on_key_up(int key);


// key code constants:

const int KEY_ENTER = 0xff0d;
const int KEY_ESC = 0xff1b;
const int KEY_TAB = 0xff09;
const int KEY_CTRL_L = 0xffe3;
const int KEY_CTRL_R = 0xffe4;
const int KEY_ALT_L = 0xffe9;
const int KEY_ALT_R = 0xffea;
const int KEY_SHIFT_L = 0xffe1;
const int KEY_SHIFT_R = 0xffe2;
const int KEY_LEFT = 0xff51;
const int KEY_RIGHT = 0xff53;
const int KEY_UP = 0xff52;
const int KEY_DOWN = 0xff54;

#endif
