/*
    Drawing shapes based on textual input, with translation implemented.

    This file is part of Simple Graphics Framework (SGF).

    Copyright 2023 Arnold Beiland

    Simple Graphics Framework is free software: you can redistribute it and/or
    modify it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along with
    this program (look for a file named COPYING in the top directory). If not, see
    <https://www.gnu.org/licenses/>.
*/

#include "../sgf.h"
#include <fstream>
#include <algorithm>
#include <cmath>
#include <cstring>
#include <cstdlib>
using namespace std;

const int MAX_SHAPES = 100;
const int MAX_TYPE_LENGTH = 20;
const int MAX_NR_DATAPOINTS = 20;

struct shape
{
    char type[MAX_TYPE_LENGTH + 1];
    Color color;
    int data[MAX_NR_DATAPOINTS];
};

struct point {
    int row, col;
};

struct line {
    // a*row + b*col + c = 0
    double a,b,c;
};

Color read_color(istream &in);
bool is_point_in_shape(point p, const shape &s);
line get_line_through_points(point p1, point p2);
bool are_points_on_same_side(line l, point p1, point p2);

int n_shapes = 0;
shape shapes[MAX_SHAPES];
int tr_row = 0;
int tr_col = 0;
bool is_mouse_down = false;
int last_mouse_row, last_mouse_col;

void initialize()
{
    ifstream fin("textdraw-input.txt");
    if (!fin) {
        logfile << "textdraw-input.txt could not be opened" << endl;
        exit(1);
    }

    fin >> n_shapes;

    for (int i = 0; i < n_shapes; ++i) {
        fin >> shapes[i].type;

        int nr_datapoints = 0;

        if (strcmp(shapes[i].type, "rectangle") == 0) {
            nr_datapoints = 4;
        }
        else if (strcmp(shapes[i].type, "circle") == 0) {
            nr_datapoints = 3;
        }
        else if (strcmp(shapes[i].type, "triangle") == 0) {
            nr_datapoints = 6;
        }
        else {
            logfile << "unknown type: " << shapes[i].type << endl;
            exit(1);
        }

        for (int j = 0; j < nr_datapoints; ++j)
            fin >> shapes[i].data[j];

        shapes[i].color = read_color(fin);
    }
}

void render(int width, int height)
{
    for (int r = 0; r < height; ++r)
        for (int c = 0; c < width; ++c) {
            point p {tr_row + r, tr_col + c};
            Color color {255, 255, 255};

            for (int i = 0; i < n_shapes; ++i)
                if (is_point_in_shape(p, shapes[i]))
                    color = shapes[i].color;

            draw_pixel(r, c, color);
        }
}


Color read_color(istream &in)
{
    int r, g, b;
    in >> r >> g >> b;

    Color color;
    color.r = r;
    color.g = g;
    color.b = b;
    return color;
}

bool is_point_in_shape(point p, const shape &s)
{
    if (strcmp(s.type, "rectangle") == 0) {
        int from_row = s.data[0];
        int from_col = s.data[1];
        int to_row = s.data[2];
        int to_col = s.data[3];

        return p.row >= from_row
            && p.row <= to_row
            && p.col >= from_col
            && p.col <= to_col;
    }
    else if (strcmp(s.type, "circle") == 0) {
        int center_row = s.data[0];
        int center_col = s.data[1];
        int radius = s.data[2];

        int d_row = p.row - center_row;
        int d_col = p.col - center_col;

        int r2 = d_row * d_row + d_col * d_col;
        return r2 <= radius * radius;
    }
    else if (strcmp(s.type, "triangle") == 0) {
        point p1, p2, p3;
        p1.row = s.data[0];
        p1.col = s.data[1];
        p2.row = s.data[2];
        p2.col = s.data[3];
        p3.row = s.data[4];
        p3.col = s.data[5];

        line side12 = get_line_through_points(p1, p2);
        line side13 = get_line_through_points(p1, p3);
        line side23 = get_line_through_points(p2, p3);

        bool is_inside = are_points_on_same_side(side12, p3, p)
            && are_points_on_same_side(side13, p2, p)
            && are_points_on_same_side(side23, p1, p);

        return is_inside;
    }

    return false;
}

line get_line_through_points(point p1, point p2)
{
    if (p1.row == p2.row)
        return { 1, 0, -1.0 * p1.row };

    double slope = (p2.col - p1.col) / (p2.row - p1.row);
    return { slope, -1.0, p1.col - p1.row*slope };
}

bool are_points_on_same_side(line l, point p1, point p2)
{
    double value1 = l.a * p1.row + l.b * p1.col + l.c;
    double value2 = l.a * p2.row + l.b * p2.col + l.c;

    return value1 * value2 > 0;
}

bool on_move(int row, int col)
{
    bool should_update = false;

    if (is_mouse_down) {
        tr_row -= row - last_mouse_row;
        tr_col -= col - last_mouse_col;
        should_update = true;
    }

    last_mouse_row = row;
    last_mouse_col = col;
    return should_update;
}

bool on_mouse_down()
{
    is_mouse_down = true;
    return true;
}

bool on_mouse_up() {
    is_mouse_down = false;
    return false;
}

bool on_scroll(int) { return false; }
bool on_key_down(int) { return false; }
bool on_key_up(int) { return false; }
