/*
    This program will let the user draw on a screen (like a pencil in a paining
    program). Fourth version - making the line continuous (works for any slope).

    This file is part of Simple Graphics Framework (SGF).

    Copyright 2023 Arnold Beiland

    Simple Graphics Framework is free software: you can redistribute it and/or
    modify it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along with
    this program (look for a file named COPYING in the top directory). If not, see
    <https://www.gnu.org/licenses/>.
*/

#include "../sgf.h"
#include <iostream>
#include <algorithm>
#include <cmath>
using namespace std;

const int MAX_ROW = 1080;
const int MAX_COL = 1920;

Color pixels[MAX_ROW][MAX_COL];
int last_mouse_row = 0, last_mouse_col = 0;
bool mouse_down = false;

int last_drawn_point_row = -1;
int last_drawn_point_col = -1;

Color drawing_color = {0,0,0};
int point_size = 3;

void draw_to(int row, int col, int continue_from_last);

void initialize()
{
    for (int i = 0; i < MAX_ROW; ++i)
        for (int j = 0; j < MAX_COL; ++j)
            pixels[i][j] = {255,255,255};
}

void render(int width, int height)
{
    for (int i = 0; i < min(height, MAX_ROW); i++)
        for (int j = 0; j < min(width, MAX_COL); j++)
            draw_pixel(i, j, pixels[i][j]);
}

bool on_move(int row, int col)
{
    last_mouse_row = row;
    last_mouse_col = col;

    if (mouse_down) {
        draw_to(row, col, true);
        return true;
    }

    return false;
}

bool on_mouse_down()
{
    mouse_down = true;
    draw_to(last_mouse_row, last_mouse_col, false);
    return true;
}

bool on_mouse_up()
{
    mouse_down = false;
    return false;
}

void draw_single_point(int row, int col) {
    int delta_low = point_size % 2 == 0 ? -point_size/2+1 : -point_size/2 ;
    int delta_high = point_size % 2 == 0 ? point_size/2 : point_size/2 ;

    for (int i = row + delta_low; i <= row + delta_high; ++i)
        for (int j = col + delta_low; j <= col + delta_high; ++j)
            if (i >= 0 && i < MAX_ROW && j >= 0 && j < MAX_COL)
                pixels[i][j] = drawing_color;
}

void draw_to(int row, int col, int continue_from_last)
{
    if (continue_from_last) {
        if (last_drawn_point_col == col) { // vertical line
            int r_start = min(last_drawn_point_row, row);
            int r_end = max(last_drawn_point_row, row);

            for (int r = r_start; r <= r_end; r++)
                draw_single_point(r, col);
        }
        else if (last_drawn_point_row == row) { // horizontal line
            int c_start = min(last_drawn_point_col, col);
            int c_end = max(last_drawn_point_col, col);

            for (int c = c_start; c <= c_end; c++)
                draw_single_point(row, c);
        }
        else if (abs(col - last_drawn_point_col) >= abs(row - last_drawn_point_row)) {
            // row = a*col + b
            double a = 1.0 * (row - last_drawn_point_row) / (col - last_drawn_point_col);
            double b = row - a * col;

            int c_start = min(last_drawn_point_col, col);
            int c_end = max(last_drawn_point_col, col);

            for (int c = c_start; c <= c_end; c++) {
                int r = round(a*c + b);
                draw_single_point(r, c);
            }
        }
        else {
            // col = a*row + b
            double a = 1.0 * (col - last_drawn_point_col) / (row - last_drawn_point_row);
            double b = col - a * row;

            int r_start = min(last_drawn_point_row, row);
            int r_end = max(last_drawn_point_row, row);

            for (int r = r_start; r <= r_end; r++) {
                int c = round(a*r + b);
                draw_single_point(r, c);
            }
        }
    }
    else
        draw_single_point(row, col);

    last_drawn_point_row = row;
    last_drawn_point_col = col;
}

bool on_key_down(int key)
{
    if (key == 'r') {
        initialize();
        return true;
    }

    return false;
}

bool on_scroll(int) { return false; }
bool on_key_up(int) { return false; }
