#include "OrdinaryBasisFunctionTests.h"
#include <cmath>
#include "../../project/Utils/DoubleEquality.h"
#include "../../project/Core/Constants.h"
#include "../../project/EC/OrdinaryBasisFunction.h"
using namespace cagd;

void OrdinaryBasisFunctionTests::initTestCase()
{
    _testValues.reserve(100);

    _testValues.push_back(0);
    _testValues.push_back(1);
    _testValues.push_back(-1);
    _testValues.push_back(PI);
    _testValues.push_back(-PI);
    _testValues.push_back(PI/2);
    _testValues.push_back(-PI/2);
    _testValues.push_back(PI/3);
    _testValues.push_back(-PI/3);

    for (int i = 1; i <= 3; ++i) {
        _testValues.push_back(std::pow(2, i));
        _testValues.push_back(- std::pow(2, i));
        _testValues.push_back(std::pow(2, -i));
        _testValues.push_back(- std::pow(2, -i));
    }
}


void OrdinaryBasisFunctionTests::identityTest()
{
    OrdinaryBasisFunction func(0, 0);

    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(func.getDerivative(0, value), 1),
            "Wrong function value");

    for (unsigned order = 1; order < 10; ++order)
        for (double value : _testValues)
            QVERIFY2(
                doubleEquals(func.getDerivative(order, value), 0),
                "Wrong derivative, order >= 1");
}

void OrdinaryBasisFunctionTests::xTest()
{
    OrdinaryBasisFunction func(1, 0);

    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(func.getDerivative(0, value), value),
            "Wrong function value");

    for (double value : _testValues)
            QVERIFY2(
                doubleEquals(func.getDerivative(1, value), 1),
                "Wrong first order derivative");

    for (unsigned order = 2; order < 10; ++order)
        for (double value : _testValues)
            QVERIFY2(
                doubleEquals(func.getDerivative(order, value), 0),
                "Wrong derivative, order >= 2");
}

void OrdinaryBasisFunctionTests::xSquaredTest()
{
    OrdinaryBasisFunction func(2, 0);

    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(func.getDerivative(0, value), value * value),
            "Wrong function value");

    for (double value : _testValues)
            QVERIFY2(
                doubleEquals(func.getDerivative(1, value), 2 * value),
                "Wrong first order derivative");

    for (double value : _testValues)
            QVERIFY2(
                doubleEquals(func.getDerivative(2, value), 2),
                "Wrong second order derivative");


    for (unsigned order = 3; order < 10; ++order)
        for (double value : _testValues)
            QVERIFY2(
                doubleEquals(func.getDerivative(order, value), 0),
                "Wrong derivative, order >= 3");
}

void OrdinaryBasisFunctionTests::expTest()
{
    OrdinaryBasisFunction func(0, 1);

    for (unsigned order = 0; order < 10; ++order)
        for (double value : _testValues)
            QVERIFY2(
                doubleEquals(func.getDerivative(order, value), std::exp(value)),
                "Wrong derivative, order >= 0");
}

void OrdinaryBasisFunctionTests::exp3xTest()
{
    OrdinaryBasisFunction func(0, 3);

    for (unsigned order = 0; order < 1; ++order)
        for (double value : _testValues)
            QVERIFY2(
                doubleEquals(func.getDerivative(order, value), std::pow(3, order) * std::exp(3 * value)),
                "Wrong derivative, order >= 0");
}

void OrdinaryBasisFunctionTests::sinTest()
{
    OrdinaryBasisFunction func(0, 0, 1, true);

    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(func.getDerivative(0, value), std::sin(value)),
            "Wrong function value");
    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(func.getDerivative(1, value), std::cos(value)),
            "Wrong 1-st order derivative");
    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(func.getDerivative(2, value), - std::sin(value)),
            "Wrong 2-nd order derivative");
    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(func.getDerivative(3, value), - std::cos(value)),
            "Wrong 3-rd order derivative");
    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(func.getDerivative(4, value), std::sin(value)),
            "Wrong 4-th order derivative");
}

void OrdinaryBasisFunctionTests::cosTest()
{
    OrdinaryBasisFunction func(0, 0, 1, false);

    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(func.getDerivative(0, value), std::cos(value)),
            "Wrong function value");
    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(func.getDerivative(1, value), - std::sin(value)),
            "Wrong 1-st order derivative");
    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(func.getDerivative(2, value), - std::cos(value)),
            "Wrong 2-nd order derivative");
    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(func.getDerivative(3, value), std::sin(value)),
            "Wrong 3-rd order derivative");
    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(func.getDerivative(4, value), std::cos(value)),
            "Wrong 4-th order derivative");
}

void OrdinaryBasisFunctionTests::xcosxTest()
{
    OrdinaryBasisFunction func(1, 0, 1, false);

    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(
                func.getDerivative(0, value),
                value * std::cos(value)),
            "Wrong function value");
    for (double value : _testValues) {
        QVERIFY2(
            doubleEquals(
                func.getDerivative(1, value),
                std::cos(value) - value * std::sin(value)),
            "Wrong 1-st order derivative");
    }
    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(
                func.getDerivative(2, value),
                - 2 * std::sin(value) - value * std::cos(value) ),
            "Wrong 2-nd order derivative");
}

void OrdinaryBasisFunctionTests::expSin2xTest()
{
    OrdinaryBasisFunction func(0, 1, 2, true);

    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(
                func.getDerivative(0, value),
                std::exp(value) * std::sin(2 * value)),
            "Wrong function value");
    for (double value : _testValues) {
        QVERIFY2(
            doubleEquals(
                func.getDerivative(1, value),
                std::exp(value) * std::sin(2 * value) + 2 * std::exp(value) * std::cos(2 * value)),
            "Wrong 1-st order derivative");
    }
    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(
                func.getDerivative(2, value),
                -3 * std::exp(value) * std::sin(2 * value) + 4 * std::exp(value) * std::cos(2 * value)),
            "Wrong 2-nd order derivative");
}

void OrdinaryBasisFunctionTests::xexp0dot1xCos3xTest()
{
    OrdinaryBasisFunction func(1, 0.1, 3, false);

    for (double value : _testValues)
        QVERIFY2(
            doubleEquals(
                func.getDerivative(0, value),
                value * std::exp(0.1 * value) * std::cos(3 * value)),
            "Wrong function value");
    for (double value : _testValues) {
        QVERIFY2(
            doubleEquals(
                func.getDerivative(1, value),
                std::exp(0.1 * value) * std::cos(3 * value)
                    + 0.1 * value * std::exp(0.1 * value) * std::cos(3 * value)
                    - 3 * value * std::exp(0.1 * value) * std::sin(3 * value)),
            "Wrong 1-st order derivative");
    }
}
