#include "benchmarking.h"
#include "EC/ECSpace.h"
#include "EC/BCurve3.h"
#include "EC/BSurface3.h"
#include <utility>
#include "Core/Constants.h"
#include <ctime>
#include <ratio>
#include <chrono>
#include <cstdlib>
#include <iostream>
using namespace std;
using namespace cagd;
using namespace std::chrono;

const int REPEAT_TIMES = 100;

double radoms[ 1<<16 ];
int crandoms = 0;

ECSpace *create_P5();
ECSpace *create_P11();
ECSpace *create_T11();
ECSpace *create_AT13();
ECSpace *create_AE10();
ECSpace *create_AET13();
ECSpace *getECSpaceForLine(int line);
void init_random();
inline double myrand();


void benchmark_curves()
{
    init_random();

    cout << "CURVES\n";

    for (int line = 0; line < 6; ++line) {
        cout << "  LINE " << line << ": ";

        steady_clock::time_point t1 = steady_clock::now();
        for (int repeat = 0; repeat < REPEAT_TIMES; ++repeat) {

            auto space = getECSpaceForLine(line);
            space->preprocessing();

            auto curve = new BCurve3(space);

            for (unsigned i = 0; i < space->getDimension(); ++i)
                (*curve)[i] = DCoordinate3(myrand(), myrand(), myrand());

            curve->GenerateImage(0, 1000);

            delete curve;
            delete space;
        }

        steady_clock::time_point t2 = steady_clock::now();

        duration<double> time_span = duration_cast<duration<double>>(t2 - t1);
        std::cout << time_span.count() << "s \n";
    }
}

void benchmark_surfaces()
{
    init_random();

    cout << "SURFACES\n";

    for (int line = 0; line < 6; ++line) {
        cout << "  LINE " << line << ": ";

        steady_clock::time_point t1 = steady_clock::now();
        for (int repeat = 0; repeat < REPEAT_TIMES; ++repeat) {

            auto spaceU = getECSpaceForLine(line);
            auto spaceV = getECSpaceForLine(line);
            spaceU->preprocessing();
            spaceV->preprocessing();

            auto surface = new BSurface3(spaceU, spaceV);

            for (unsigned i = 0; i < spaceU->getDimension(); ++i)
                 for (unsigned j = 0; j < spaceV->getDimension(); ++j)
                    (*surface)(i,j) = DCoordinate3(myrand(), myrand(), myrand());

            surface->GenerateImage(50, 50);

            delete surface;
            delete spaceV;
            delete spaceU;
        }

        steady_clock::time_point t2 = steady_clock::now();

        duration<double> time_span = duration_cast<duration<double>>(t2 - t1);
        std::cout << time_span.count() << "s \n";
    }
}


void init_random()
{
    srand((unsigned) time(NULL));
    for (int i=0; i<(1<<16); ++i)
        radoms[i] = 20.0 * rand() / RAND_MAX - 10.0;
}

double myrand()
{
    crandoms = (13 * crandoms + 39) % (1<<16);
    return radoms[crandoms];
}

ECSpace *getECSpaceForLine(int line)
{
    switch (line) {
        case 0: return create_P5();
        case 1: return create_P11();
        case 2: return create_T11();
        case 3: return create_AT13();
        case 4: return create_AE10();
        case 5: return create_AET13();
    }

    return nullptr;
}


ECSpace *create_P5()
{
    auto result = new ECSpace();
    result->definitionDomain = make_pair(0,1);

    auto &zeros = result->characteristicPolynomial.zeros;

    zeros.resize(1);
    zeros[0].real = 0;
    zeros[0].absImaginary = 0;
    zeros[0].multiplicity = 6;

    return result;
}

ECSpace *create_P11()
{
    auto result = new ECSpace();
    result->definitionDomain = make_pair(0,1);

    auto &zeros = result->characteristicPolynomial.zeros;

    zeros.resize(1);
    zeros[0].real = 0;
    zeros[0].absImaginary = 0;
    zeros[0].multiplicity = 10;

    return result;
}

ECSpace *create_T11()
{
    auto result = new ECSpace();
    result->definitionDomain = make_pair(0,PI/2);

    auto &zeros = result->characteristicPolynomial.zeros;

    zeros.resize(6);

    zeros[0].real = 0;
    zeros[0].absImaginary = 0;
    zeros[0].multiplicity = 1;

    zeros[1].real = 0;
    zeros[1].absImaginary = 1;
    zeros[1].multiplicity = 1;

    zeros[2].real = 0;
    zeros[2].absImaginary = 2;
    zeros[2].multiplicity = 1;

    zeros[3].real = 0;
    zeros[3].absImaginary = 3;
    zeros[3].multiplicity = 1;

    zeros[4].real = 0;
    zeros[4].absImaginary = 4;
    zeros[4].multiplicity = 1;

    zeros[5].real = 0;
    zeros[5].absImaginary = 5;
    zeros[5].multiplicity = 1;

    return result;
}

ECSpace *create_AT13()
{
    auto result = new ECSpace();
    result->definitionDomain = make_pair(0,PI/2);

    auto &zeros = result->characteristicPolynomial.zeros;

    zeros.resize(3);

    zeros[0].real = 0;
    zeros[0].absImaginary = 0;
    zeros[0].multiplicity = 1;

    zeros[1].real = 0;
    zeros[1].absImaginary = 1;
    zeros[1].multiplicity = 3;

    zeros[2].real = 0;
    zeros[2].absImaginary = 2;
    zeros[2].multiplicity = 3;

    return result;
}

ECSpace *create_AE10()
{
    auto result = new ECSpace();
    result->definitionDomain = make_pair(0,1);

    auto &zeros = result->characteristicPolynomial.zeros;

    zeros.resize(4);

    zeros[0].real = 0;
    zeros[0].absImaginary = 0;
    zeros[0].multiplicity = 1;

    zeros[1].real = 1;
    zeros[1].absImaginary = 0;
    zeros[1].multiplicity = 3;

    zeros[2].real = 2;
    zeros[2].absImaginary = 0;
    zeros[2].multiplicity = 3;

    zeros[3].real = 3;
    zeros[3].absImaginary = 0;
    zeros[3].multiplicity = 3;


    return result;
}

ECSpace *create_AET13()
{
    auto result = new ECSpace();
    result->definitionDomain = make_pair(0,1);

    auto &zeros = result->characteristicPolynomial.zeros;

    zeros.resize(4);

    zeros[0].real = 0;
    zeros[0].absImaginary = 0;
    zeros[0].multiplicity = 1;

    zeros[1].real = 1;
    zeros[1].absImaginary = 1;
    zeros[1].multiplicity = 2;

    zeros[2].real = 2;
    zeros[2].absImaginary = 1;
    zeros[2].multiplicity = 2;

    zeros[3].real = 1;
    zeros[3].absImaginary = 2;
    zeros[3].multiplicity = 2;

    return result;
}
