#include "ViewTab.h"
#include "ui_ViewTab.h"
#include "CommonUITools.h"
#include "Model/CurveItem.h"
#include "Model/SurfaceItem.h"

#include <QDoubleSpinBox>
#include <QColor>
#include <QColorDialog>
#include <QFileDialog>
#include <fstream>

namespace cagd
{
    ViewTab::ViewTab(QWidget *parent, Scene *scene) :
        QWidget(parent),
        ui(new Ui::ViewTab),
        _scene(scene)
    {
        ui->setupUi(this);

        connect(ui->lineWidthSpinBox, SIGNAL(valueChanged(double)), this, SLOT(lineWidthChanged(double)));
        connect(ui->selectionLineWidthSpinBox, SIGNAL(valueChanged(double)),
            this, SLOT(selectionLineWidthChanged(double)));
        connect(ui->curveDivPointCountSpinBox, SIGNAL(valueChanged(int)), this, SLOT(curveDivPointCountChanged(int)));
        connect(ui->surfaceDivPointCountSpinBox, SIGNAL(valueChanged(int)),
            this, SLOT(surfaceDivPointCountChanged(int)));
        connect(ui->isoLineCountSpinBox, SIGNAL(valueChanged(int)), this, SLOT(isoLineCountChanged(int)));
        connect(ui->setControlLinesColorButton, SIGNAL(clicked()), this, SLOT(setControlLinesColor()));
        connect(ui->setBackgroundColorButton, SIGNAL(clicked()), this, SLOT(setBackgroundColor()));
        connect(ui->addCurveButton, SIGNAL(clicked()), this, SLOT(addDummyCurve()));
        connect(ui->loadCurveButton, SIGNAL(clicked()), this, SLOT(loadCurveFromFile()));
        connect(ui->addSurfaceButton, SIGNAL(clicked()), this, SLOT(addDummySurface()));
        connect(ui->loadSurfaceButton, SIGNAL(clicked()), this, SLOT(loadSurfaceFromFile()));
        connect(ui->saveConfigurationButton, SIGNAL(clicked()), this, SLOT(saveConfigurationToFile()));
        connect(ui->loadConfigurationButton, SIGNAL(clicked()), this, SLOT(loadConfigurationFromFile()));
        connect(ui->defaultViewButton, SIGNAL(clicked()), this, SIGNAL(setDefaultView()));
        connect(ui->centerSelectionButton, SIGNAL(clicked()), this, SIGNAL(moveSelectionToCenter()));
    }

    ViewTab::~ViewTab()
    {
        delete ui;
    }

    void ViewTab::updateFromScene()
    {
        ui->lineWidthSpinBox->setValue(_scene->lineWidth);
        ui->selectionLineWidthSpinBox->setValue(_scene->lineWidthForSelection);
        ui->curveDivPointCountSpinBox->setValue(_scene->curveDivPointCount);
        ui->surfaceDivPointCountSpinBox->setValue(_scene->surfaceDivPointCount);
        ui->isoLineCountSpinBox->setValue(_scene->isoparametricLineCount);
        ui->centerSelectionButton->setEnabled(_scene->selection.selectedItem);
    }


    void ViewTab::lineWidthChanged(double value)
    {
        if (std::abs(_scene->lineWidth - (float) value) >= 0.001) {
            _scene->lineWidth = (float) value;
            configChanged();
        }
    }

    void ViewTab::selectionLineWidthChanged(double value)
    {
        if (std::abs(_scene->lineWidthForSelection - (float) value) >= 0.001) {
            _scene->lineWidthForSelection = (float) value;
            configChanged();
        }
    }

    void ViewTab::curveDivPointCountChanged(int value)
    {
        if (_scene->curveDivPointCount != value) {
            _scene->curveDivPointCount = value;
            for (auto item : _scene->items)
                if (dynamic_cast<CurveItem *>(item))
                    item->update();
            configChanged();
        }
    }

    void ViewTab::surfaceDivPointCountChanged(int value)
    {
        if (_scene->surfaceDivPointCount != value) {
            _scene->surfaceDivPointCount = value;
            for (auto item : _scene->items)
                if (dynamic_cast<SurfaceItem *>(item))
                    item->update();
            configChanged();
        }
    }

    void ViewTab::isoLineCountChanged(int value)
    {
        if (_scene->isoparametricLineCount != value) {
            _scene->isoparametricLineCount = value;
            for (auto item : _scene->items)
                if (dynamic_cast<SurfaceItem *>(item))
                    item->update();
            configChanged();
        }
    }

    void ViewTab::setControlLinesColor()
    {
        if (CommonUITools::changeColorViaPopup(_scene->controlLinesColor))
            configChanged();
    }

    void ViewTab::setBackgroundColor()
    {
        if (CommonUITools::changeColorViaPopup(_scene->backgroundColor))
            configChanged();
    }

    void ViewTab::addDummyCurve()
    {
        auto item = new CurveItem(_scene);

        item->ecSpace = new ECSpace();
        item->ecSpace->definitionDomain = std::make_pair(0,1);

        item->ecSpace->characteristicPolynomial.zeros.resize(1);
        item->ecSpace->characteristicPolynomial.zeros[0].real = 0;
        item->ecSpace->characteristicPolynomial.zeros[0].absImaginary = 0;
        item->ecSpace->characteristicPolynomial.zeros[0].multiplicity = 3;

        item->ecSpace->preprocessing();

        item->bcurve = new BCurve3(item->ecSpace);

        (*item->bcurve)[0] = DCoordinate3(
            0.5 * std::rand() / RAND_MAX,
            0.5 * std::rand() / RAND_MAX,
            0.5 * std::rand() / RAND_MAX);

        for (unsigned i = 1; i < item->bcurve->GetControlPointCount(); ++i)
            (*item->bcurve)[i] = (*item->bcurve)[i-1] + DCoordinate3(
                2.0 * std::rand() / RAND_MAX,
                2.0 * std::rand() / RAND_MAX,
                2.0 * std::rand() / RAND_MAX);

        item->update();
        _scene->items.push_back(item);
        _scene->selection.selectedItem = item;
        _scene->selection.selectedPoint = nullptr;
        _scene->shouldTriggerTabChange = true;
        configChanged();
    }

    void ViewTab::loadCurveFromFile()
    {
        auto fileName = QFileDialog::getOpenFileName(
            this,
            "Select File to Open",
            "");

        if (fileName.length() > 0)
        {
            std::ifstream fin(fileName.toUtf8().constData());
            auto item = new CurveItem(_scene);
            fin >> *item;
            _scene->items.push_back(item);
            _scene->selection.selectedItem = item;
            _scene->selection.selectedPoint = nullptr;
            _scene->shouldTriggerTabChange = true;
            configChanged();
        }
    }

    void ViewTab::addDummySurface()
    {
        SurfaceItem *item = new SurfaceItem(_scene);

        item->ecSpaceU = new ECSpace();
        item->ecSpaceU->definitionDomain = std::make_pair(0, 1);
        item->ecSpaceU->characteristicPolynomial.zeros[0].multiplicity = 3;
        item->ecSpaceU->preprocessing();

        item->ecSpaceV = new ECSpace(*item->ecSpaceU);
        item->ecSpaceV->preprocessing();

        item->bsurface = new BSurface3(item->ecSpaceU, item->ecSpaceV);
        (*item->bsurface)(0,0) = DCoordinate3(0, 2, 0);
        (*item->bsurface)(0,1) = DCoordinate3(1, 2, 0);
        (*item->bsurface)(0,2) = DCoordinate3(2, 2, 0);

        (*item->bsurface)(1,0) = DCoordinate3(0, 1, 0);
        (*item->bsurface)(1,1) = DCoordinate3(1, 1, 4);
        (*item->bsurface)(1,2) = DCoordinate3(2, 1, 0);

        (*item->bsurface)(2,0) = DCoordinate3(0, 0, 0);
        (*item->bsurface)(2,1) = DCoordinate3(1, 0, 0);
        (*item->bsurface)(2,2) = DCoordinate3(2, 0, 0);

        item->update();

        _scene->items.push_back(item);
        _scene->selection.selectedItem = item;
        _scene->selection.selectedPoint = nullptr;
        _scene->shouldTriggerTabChange = true;
        configChanged();
    }

    void ViewTab::loadSurfaceFromFile()
    {
        auto fileName = QFileDialog::getOpenFileName(
            this,
            "Select File to Open",
            "");

        if (fileName.length() > 0)
        {
            std::ifstream fin(fileName.toUtf8().constData());
            auto item = new SurfaceItem(_scene);
            fin >> *item;
            _scene->items.push_back(item);
            _scene->selection.selectedItem = item;
            _scene->selection.selectedPoint = nullptr;
            _scene->shouldTriggerTabChange = true;
            configChanged();
        }
    }

    void ViewTab::loadConfigurationFromFile()
    {
        auto fileName = QFileDialog::getOpenFileName(
            this,
            "Select File to Open",
            "");

        if (fileName.length() > 0)
        {
            std::ifstream fin(fileName.toUtf8().constData());

            for (auto item : _scene->items)
                delete item;

            _scene->items.clear();

            int count;
            fin >> count;

            for (int i = 0; i < count; ++i) {
                std::string type;
                fin >> type;

                if (type == "curve") {
                   auto curve = new CurveItem(_scene);
                   fin >> *curve;
                   _scene->items.push_back(curve);
                }
                else {
                    auto surface = new SurfaceItem(_scene);
                    fin >> *surface;
                    _scene->items.push_back(surface);
                }
            }

            _scene->selection.selectedItem = nullptr;
            _scene->selection.selectedPoint = nullptr;
            configChanged();
        }
    }

    void ViewTab::saveConfigurationToFile()
    {
        auto fileName = QFileDialog::getSaveFileName(
            this,
            "Select File to Save",
            "");

        if (fileName.length() > 0)
        {
            std::ofstream fout(fileName.toUtf8().constData());
            fout << _scene->items.size() << std::endl;

            for (auto item : _scene->items) {
                auto curve = dynamic_cast<CurveItem *>(item);
                auto surface = dynamic_cast<SurfaceItem *>(item);

                if (curve) {
                    fout << "curve\n" << *curve << std::endl;
                }

                if (surface) {
                    fout << "surface\n" << *surface << std::endl;
                }
            }
        }
    }
}
