#include "SurfaceSetupTab.h"
#include "ui_SurfaceSetupTab.h"
#include "CommonUITools.h"
#include "SetupECSpaces/SetupECSpacesDialog.h"
#include "Model/SurfaceItem.h"
#include "SetupExactDescription/SetupExactDescriptionDialog.h"

#include <QFileDialog>
#include <QDoubleSpinBox>
#include <QPushButton>
#include <QToolButton>
#include <QLabel>
#include <iomanip>
#include <QCheckBox>
#include <QComboBox>
#include <fstream>

namespace cagd
{
    SurfaceSetupTab::SurfaceSetupTab(QWidget *parent, Scene *scene) :
        QWidget(parent),
        ui(new Ui::SurfaceSetupTab),
        _scene(scene)
    {
        ui->setupUi(this);

        connect(ui->alwaysShowControlNetCheckBox, SIGNAL(toggled(bool)),
            this, SLOT(alwaysShowControlNetChanged(bool)));
        connect(ui->showNormalsCheckBox, SIGNAL(toggled(bool)), this, SLOT(showNormalsChanged(bool)));
        connect(ui->showIsoLinesCheckBox, SIGNAL(toggled(bool)), this, SLOT(showIsoLinesChanged(bool)));
        connect(ui->setNormalsColorButtton, SIGNAL(clicked()), this, SLOT(setNormalsColor()));
        connect(ui->setIsoLinesColorButton, SIGNAL(clicked()), this, SLOT(setIsoLinesColor()));
        connect(ui->materialCombo, SIGNAL(currentIndexChanged(int)), this, SLOT(materialChanged(int)));
        connect(ui->deleteSurfaceButton, SIGNAL(clicked()), this, SLOT(deleteSurface()));
        connect(ui->setupECSpaceUButton, SIGNAL(clicked()), this, SLOT(setupECSpaceUClicked()));
        connect(ui->setupECSpaceVButton, SIGNAL(clicked()), this, SLOT(setupECSpaceVClicked()));
        connect(ui->exactDescriptionButton, SIGNAL(clicked()), this, SLOT(exactDescriptionClicked()));
        connect(ui->uOrderElevationButton, SIGNAL(clicked()), this, SLOT(performOrderElevationU()));
        connect(ui->vOrderElevationButton, SIGNAL(clicked()), this, SLOT(performOrderElevationV()));
        connect(ui->uSubdivisionButton, SIGNAL(clicked()), this, SLOT(performUSubdivision()));
        connect(ui->vSubdivisionButton, SIGNAL(clicked()), this, SLOT(performVSubdivision()));
        connect(ui->saveSurfaceButton, SIGNAL(clicked()), this, SLOT(saveSurfaceToFile()));
    }

    SurfaceSetupTab::~SurfaceSetupTab()
    {
        delete ui;
    }

    void SurfaceSetupTab::updateFromScene()
    {
        auto item = dynamic_cast<SurfaceItem *>(_scene->selection.selectedItem);

        ui->alwaysShowControlNetCheckBox->setChecked(item->alwaysShowControlNet);
        ui->showNormalsCheckBox->setChecked(item->showNormals);
        ui->showIsoLinesCheckBox->setChecked(item->showIsoparametricLines);

        auto &materials = CommonUITools::listOfMaterials;
        int currentMaterialIndex = (int) (
            std::find(materials.begin(), materials.end(), item->material) -
            materials.begin());
        ui->materialCombo->setCurrentIndex(currentMaterialIndex);

        auto uInterval = item->ecSpaceU->definitionDomain;
        ui->uIntervalLeftLabel->setText(formatDouble(uInterval.first).c_str());
        ui->uIntervalRightLabel->setText(formatDouble(uInterval.second).c_str());

        ui->uSubdivisionPointSpinBox->setMinimum(uInterval.first + 0.0001);
        ui->uSubdivisionPointSpinBox->setMaximum(uInterval.second - 0.0001);
        ui->uSubdivisionPointSpinBox->setValue((uInterval.first + uInterval.second) / 2);

        auto vInterval = item->ecSpaceV->definitionDomain;
        ui->vIntervalLeftLabel->setText(formatDouble(vInterval.first).c_str());
        ui->vIntervalRightLabel->setText(formatDouble(vInterval.second).c_str());

        ui->vSubdivisionPointSpinBox->setMinimum(vInterval.first + 0.0001);
        ui->vSubdivisionPointSpinBox->setMaximum(vInterval.second - 0.0001);
        ui->vSubdivisionPointSpinBox->setValue((vInterval.first + vInterval.second) / 2);
    }

    void SurfaceSetupTab::alwaysShowControlNetChanged(bool value)
    {
        auto item = dynamic_cast<SurfaceItem *>(_scene->selection.selectedItem);

        if (item->alwaysShowControlNet != value) {
            item->alwaysShowControlNet = value;
            configChanged();
        }
    }

    void SurfaceSetupTab::showNormalsChanged(bool value)
    {
        auto item = dynamic_cast<SurfaceItem *>(_scene->selection.selectedItem);

        if (item->showNormals != value) {
            item->showNormals = value;
            configChanged();
        }
    }

    void SurfaceSetupTab::showIsoLinesChanged(bool value)
    {
        auto item = dynamic_cast<SurfaceItem *>(_scene->selection.selectedItem);

        if (item->showIsoparametricLines != value) {
            item->showIsoparametricLines = value;
            configChanged();
        }
    }

    void SurfaceSetupTab::setNormalsColor()
    {
        auto item = dynamic_cast<SurfaceItem *>(_scene->selection.selectedItem);

        if (CommonUITools::changeColorViaPopup(item->normalsColor))
            configChanged();
    }

    void SurfaceSetupTab::setIsoLinesColor()
    {
        auto item = dynamic_cast<SurfaceItem *>(_scene->selection.selectedItem);

        if (CommonUITools::changeColorViaPopup(item->isoLinesColor))
            configChanged();
    }

    void SurfaceSetupTab::materialChanged(int index)
    {
        auto item = dynamic_cast<SurfaceItem *>(_scene->selection.selectedItem);
        auto &materials = CommonUITools::listOfMaterials;
        int currentIndex = (int) (
            std::find(materials.begin(), materials.end(), item->material) -
            materials.begin());

        if (currentIndex != index) {
            item->material = materials[index];
            configChanged();
        }
    }

    void SurfaceSetupTab::setupECSpaceUClicked()
    {
        auto item = dynamic_cast<SurfaceItem *>(_scene->selection.selectedItem);

        auto dialog = new SetupECSpacesDialog(
            nullptr,
            _scene->ecSpaceDatabase,
            item->ecSpaceU);

        bool result = dialog->exec();
        delete dialog;
        activateMainGLWidget();

        if (result)
        {
            item->regenerate();
            configChanged();
        }
    }

    void SurfaceSetupTab::setupECSpaceVClicked()
    {
        auto item = dynamic_cast<SurfaceItem *>(_scene->selection.selectedItem);

        auto dialog = new SetupECSpacesDialog(
            nullptr,
            _scene->ecSpaceDatabase,
            item->ecSpaceV);

        bool result = dialog->exec();
        delete dialog;
        activateMainGLWidget();

        if (result)
        {
            item->regenerate();
            configChanged();
        }
    }

    void SurfaceSetupTab::exactDescriptionClicked()
    {
         auto item = dynamic_cast<SurfaceItem *>(_scene->selection.selectedItem);
        if (!item) return;

        std::vector<DCoordinate3> result;
        SetupExactDescriptionDialog dialog(nullptr, &result, item->ecSpaceU, item->ecSpaceV);

        if (dialog.exec() == QDialog::Accepted) {
            item->updateControlPointsForExactDescription(result);
        }
    }

    void SurfaceSetupTab::performOrderElevationU()
    {
        auto item = dynamic_cast<SurfaceItem *>(_scene->selection.selectedItem);

        item->performOrderElevationU(
            ui->zeroRealSpinBox->value(),
            ui->zeroImagSpinBox->value(),
            ui->multiplicitySpinBox->value());
        _scene->selection.selectedPoint = nullptr;
        configChanged();
    }

    void SurfaceSetupTab::performOrderElevationV()
    {
        auto item = dynamic_cast<SurfaceItem *>(_scene->selection.selectedItem);

        item->performOrderElevationV(
            ui->zeroRealSpinBox->value(),
            ui->zeroImagSpinBox->value(),
            ui->multiplicitySpinBox->value());
        _scene->selection.selectedPoint = nullptr;
        configChanged();
    }

    void SurfaceSetupTab::performUSubdivision()
    {
        auto item = dynamic_cast<SurfaceItem *>(_scene->selection.selectedItem);

        item->performSubdivisionU(ui->uSubdivisionPointSpinBox->value());
        _scene->selection.selectedPoint = nullptr;
        configChanged();
    }

    void SurfaceSetupTab::performVSubdivision()
    {
        auto item = dynamic_cast<SurfaceItem *>(_scene->selection.selectedItem);

        item->performSubdivisionV(ui->vSubdivisionPointSpinBox->value());
        _scene->selection.selectedPoint = nullptr;
        configChanged();
    }

    std::string SurfaceSetupTab::formatDouble(double value) const
    {
        std::stringstream ss;
        ss << std::fixed << std::setprecision(4) << value;
        return ss.str();
    }


    void SurfaceSetupTab::deleteSurface()
    {
        _scene->deleteItem(_scene->selection.selectedItem);
        configChanged();
    }

    void SurfaceSetupTab::saveSurfaceToFile()
    {
        auto item = dynamic_cast<SurfaceItem *>(_scene->selection.selectedItem);

        auto fileName = QFileDialog::getSaveFileName(
            this,
            "Select File to Save",
            "");

        if (fileName.length() > 0) {
            std::ofstream fout(fileName.toUtf8().constData());
            fout << *item;
        }
    }


}
