#include "SetupExactDescriptionDialog.h"
#include "ui_SetupExactDescriptionDialog.h"
#include "BasisColumn.h"
#include "../../Utils/DoubleEquality.h"

#include <QHBoxLayout>
#include <QFileDialog>
#include <QPushButton>
#include <QScrollArea>
#include <QLabel>
#include <iomanip>
#include <fstream>

namespace cagd
{
    SetupExactDescriptionDialog::SetupExactDescriptionDialog(
            QWidget *parent,
            std::vector<DCoordinate3> *resultVec,
            ECSpace *ecSpaceU,
            ECSpace *ecSpaceV) :
            QDialog(parent),
            ui(new Ui::SetupExactDescriptionDialog),
            _ecSpaceU(ecSpaceU),
            _ecSpaceV(ecSpaceV),
            _result(*resultVec)
    {
        ui->setupUi(this);
        ui->invisibleDummy->setVisible(false);

        connect(ui->saveToFileButton, SIGNAL(clicked()), this, SLOT(saveToFile()));
        connect(ui->loadFromFileButton, SIGNAL(clicked()), this, SLOT(loadFromFile()));

        ui->scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
        ui->scrollAreaWidgetContents->setStyleSheet("background-color: #f5f5f5;");
        ui->scrollAreaWidgetContents->setLayout(new QHBoxLayout());
        ui->scrollAreaWidgetContents->layout()->setAlignment(Qt::AlignLeft);
        ui->scrollAreaWidgetContents->layout()->setContentsMargins(0, 0, 0, 0);
        ui->scrollAreaWidgetContents->layout()->setSpacing(0);

        _columnCount =
            _ecSpaceU->getDimension() *
            (_ecSpaceV == nullptr ? 1 : _ecSpaceV->getDimension());

        _result.resize(_columnCount);
        for (auto &coord : _result)
            coord.x() = coord.y() = coord.z() = 0;

        _summaryLabel = new QLabel(this);
        _summaryLabel->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Preferred);
        ui->summaryScrollAreaContents->setStyleSheet("background-color: #f5f5f5;");
        ui->summaryScrollAreaContents->setLayout(new QHBoxLayout());
        ui->summaryScrollAreaContents->layout()->setContentsMargins(9, 3, 9, 3);
        ui->summaryScrollAreaContents->layout()->setAlignment(Qt::AlignTop);
        ui->summaryScrollAreaContents->layout()->addWidget(_summaryLabel);

        addColumns();
        generateSummary();
    }

    SetupExactDescriptionDialog::~SetupExactDescriptionDialog()
    {
        delete ui;
    }

    void SetupExactDescriptionDialog::addColumns()
    {
        for (unsigned i = 0; i < _columnCount; ++i) {
            auto text = "<span style=\"font-size: 12pt;\">" + textForColumn(i) + "</span>";
            auto column = new BasisColumn(this, &_result[i], text);
            ui->scrollAreaWidgetContents->layout()->addWidget(column);
            connect(column, SIGNAL(tripletChanged()), this, SLOT(generateSummary()));
            connect(this, SIGNAL(updateColumns()), column, SLOT(updateValues()));
        }
    }

    std::string SetupExactDescriptionDialog::textForColumn(int index)
    {
        if (_ecSpaceV == nullptr) {
            return _ecSpaceU->ordBases[index].getHTMLRepresentation("u");
        }

        int uindex = index / _ecSpaceV->getDimension();
        int vindex = index % _ecSpaceV->getDimension();

        auto utext = _ecSpaceU->ordBases[uindex].getHTMLRepresentation("u");
        auto vtext = _ecSpaceV->ordBases[vindex].getHTMLRepresentation("v");

        if (utext == "1")
            return vtext;

        if (vtext == "1")
            return utext;

        return utext + " · " + vtext;
    }

    void SetupExactDescriptionDialog::generateSummary()
    {
        std::stringstream ss;
        bool anythingWritten;

        ss << "<html><head/><body><p style=\"font-size:12pt;\">";

        ss << "x = ";
        anythingWritten = false;

        for (unsigned i = 0; i < _columnCount; ++i) {
            auto result = writeSummaryColumn(ss, i, _result[i].x(), anythingWritten);
            anythingWritten = anythingWritten || result;
        }

        if (!anythingWritten) ss << "0";

        ss << "</p><p style=\"font-size:12pt;\">";

        ss << "y = ";
        anythingWritten = false;

        for (unsigned i = 0; i < _columnCount; ++i) {
            auto result = writeSummaryColumn(ss, i, _result[i].y(), anythingWritten);
            anythingWritten = anythingWritten || result;
        }

        if (!anythingWritten) ss << "0";

        ss << "</p><p style=\"font-size:12pt;\">";

        ss << "z = ";
        anythingWritten = false;

        for (unsigned i = 0; i < _columnCount; ++i) {
            auto result = writeSummaryColumn(ss, i, _result[i].z(), anythingWritten);
            anythingWritten = anythingWritten || result;
        }

        if (!anythingWritten) ss << "0";

        ss << "</p></body></html>";
        _summaryLabel->setText(ss.str().c_str());
    }

    bool SetupExactDescriptionDialog::writeSummaryColumn(
        std::stringstream &ss, int columnIndex, double value, bool wasAnythingWritten)
    {
        if (doubleEquals(value, 0)) return false;

        auto func = textForColumn(columnIndex);

        if (value < 0) {
            value = -value;
            if(wasAnythingWritten)
                ss << "  &nbsp;-&nbsp;  ";
            else
                ss << "-  ";
        }
        else if (wasAnythingWritten)
            ss << "  &nbsp;+&nbsp;  ";

        if (doubleEquals(value, 1)) {
            ss << func;
        }
        else {
            ss << formatTrimmed(value);
            if (func != "1")
                ss << " · " << func;
        }

        return true;
    }

    std::string SetupExactDescriptionDialog::formatTrimmed(double value)
    {
        std::stringstream ss;
        ss << std::fixed << std::setprecision(6) << value;
        std::string result = ss.str();

        int nrTrailingChars = 0;
        for (int i = (int) result.size() - 1; i > 0; --i)
            if (result[i] == '0')
                ++nrTrailingChars;
            else if (result[i] == '.') {
                ++nrTrailingChars;
                break;
            }
            else break;

        return result.substr(0, result.size() - nrTrailingChars);
    }

    void SetupExactDescriptionDialog::saveToFile()
    {
        auto fileName = QFileDialog::getSaveFileName(
            this,
            "Select File to Save",
            "");

        if (fileName.length() > 0) {
            std::ofstream fout(fileName.toUtf8().constData());
            for (auto &coord : _result)
                fout << coord << std::endl;
        }
    }

    void SetupExactDescriptionDialog::loadFromFile()
    {
        auto fileName = QFileDialog::getOpenFileName(
            this,
            "Select File to Open",
            "");

        if (fileName.length() > 0)
        {
            std::ifstream fin(fileName.toUtf8().constData());
            for (auto &coord : _result)
                fin >> coord;
            updateColumns();
            generateSummary();
        }
    }
}
