#include "SetupECSpacesDialog.h"
#include "ui_SetupECSpacesDialog.h"
#include "ZeroItemWidget.h"
#include "SaveForLaterUseDialog.h"
#include "LoadExistingSpaceDialog.h"
#include "PreviewECWidget.h"

#include <iostream>
#include <QLabel>
#include <QSpinBox>
#include <QString>
#include <QAbstractItemModel>
#include <QStandardItemModel>
#include <QStringListModel>
#include <QTableView>
#include <QMessageBox>
#include <QListWidget>
#include <QGroupBox>
#include <QVBoxLayout>
#include <QScrollArea>
#include <QPushButton>
#include <vector>
#include <utility>

namespace cagd
{

    SetupECSpacesDialog::SetupECSpacesDialog(QWidget *parent, ECSpaceDatabase *database, ECSpace *originalSpace):
        QDialog(parent),
        ui(new Ui::SetupECSpacesDialog),
        _database(database)
    {
        ui->setupUi(this);

        connect(ui->editorTabs, SIGNAL(currentChanged(int)), this, SLOT(tabChanged(int)));
        connect(ui->domainLeftSpinBox, SIGNAL(valueChanged(double)), this, SLOT(domainLeftChanged(double)));
        connect(ui->domainRightSpinBox, SIGNAL(valueChanged(double)), this, SLOT(domainRightChanged(double)));
        connect(ui->addNewZeroButton, SIGNAL(clicked()), this, SLOT(addNewZeroItem()));
        connect(ui->undoChangesButton, SIGNAL(clicked()), this, SLOT(undoChangesPressed()));
        connect(ui->saveForLaterUseButton, SIGNAL(clicked()), this, SLOT(saveForLaterUsePressed()));
        connect(ui->loadExistingSetupButton, SIGNAL(clicked()), this, SLOT(loadExistingSetupPressed()));
        connect(ui->saveButton, SIGNAL(clicked()), this, SLOT(savePressed()));

        ui->domainValidation->setToolTip("Left endpoint must be less than right endpoint.");
        ui->invisibleButton->setDefault(true);
        ui->invisibleButton->setVisible(false);
        ui->previewTab->setLayout(new QVBoxLayout());
        ui->previewTab->layout()->setMargin(0);

        _zerosLayout = new QVBoxLayout();
        _zerosLayout->setAlignment(Qt::AlignTop);
        ui->zerosScrollArea->setLayout(_zerosLayout);

        _originalSpace = originalSpace;
        _editedSpace = new ECSpace(*_originalSpace);
        loadEditorData();
    }

    SetupECSpacesDialog::~SetupECSpacesDialog()
    {
        delete ui;
        delete _zerosLayout;
        delete _editedSpace;
    }

    void SetupECSpacesDialog::loadEditorData()
    {
        ui->domainLeftSpinBox->setValue(_editedSpace->definitionDomain.first);
        ui->domainRightSpinBox->setValue(_editedSpace->definitionDomain.second);

        QLayoutItem* item;
        while ((item = _zerosLayout->takeAt(0)) != nullptr)
        {
            delete item->widget();
            delete item;
        }

        for (auto zero : _editedSpace->characteristicPolynomial.zeros)
            insertZero(
                zero,
                doubleEquals(zero.real, 0) && doubleEquals(zero.absImaginary, 0));

        updateAndValidateZeros();
        changesWereMade();
        validateDomain();
    }

    void SetupECSpacesDialog::tabChanged(int which)
    {
        if (which == 1) {
            if (!_allValidationOk) {
                ui->editorTabs->setCurrentIndex(0);
                QMessageBox::critical(this, "Error", "Unable to preview due to validation errors!");
                return;
            }
            if (_previewWidget) return;

            _previewWidget = new PreviewECWidget(this, _editedSpace);
            ui->previewTab->layout()->addWidget(_previewWidget);

            ui->loadExistingSetupButton->setVisible(false);
            ui->saveForLaterUseButton->setVisible(false);
            ui->undoChangesButton->setVisible(false);
        }
        else {
            if (_previewWidget){
                ui->previewTab->layout()->takeAt(0);
                _previewWidget->setParent(nullptr);
                delete _previewWidget;
                _previewWidget = nullptr;
            }
            ui->loadExistingSetupButton->setVisible(true);
            ui->saveForLaterUseButton->setVisible(true);
            ui->undoChangesButton->setVisible(true);
        }
    }

    void SetupECSpacesDialog::domainLeftChanged(double value)
    {
        _editedSpace->definitionDomain.first = value;
        changesWereMade();
        validateDomain();
    }

    void SetupECSpacesDialog::domainRightChanged(double value)
    {
        _editedSpace->definitionDomain.second = value;
        changesWereMade();
        validateDomain();
    }

    void SetupECSpacesDialog::addNewZeroItem()
    {
        insertZero();
    }

    void SetupECSpacesDialog::insertZero(CharacteristicPolynomial::Zero zero, bool setPermanentZero)
    {
        auto newZero = new ZeroItemWidget(ui->scrollArea, zero);
        if (setPermanentZero)
            newZero->setPermanentZero();

        _zerosLayout->addWidget(newZero);
        connect(newZero, SIGNAL(deletionRequested(QWidget*)), this, SLOT(deleteZeroItem(QWidget*)));
        connect(newZero, SIGNAL(changesWereMade()), this, SLOT(updateAndValidateZeros()));

        updateAndValidateZeros();
    }

    void SetupECSpacesDialog::deleteZeroItem(QWidget *widget)
    {
        _zerosLayout->removeWidget(widget);
        delete widget;
        updateAndValidateZeros();
    }

    void SetupECSpacesDialog::validateDomain()
    {
        bool valid = _editedSpace->definitionDomain.first < _editedSpace->definitionDomain.second;
        ui->domainValidation->setVisible(!valid);
        _domainValidationOk = valid;
        propagateValidationResult();
    }

    void SetupECSpacesDialog::updateAndValidateZeros()
    {
        auto &zeros = _editedSpace->characteristicPolynomial.zeros;
        zeros.resize(_zerosLayout->count());
        _zerosValidationOk = true;

        for (int i = 0; i < _zerosLayout->count(); ++i)
        {
            auto item = _zerosLayout->itemAt(i);
            auto widget = static_cast<ZeroItemWidget *>(item->widget());

            zeros[i].real = widget->getReal();
            zeros[i].absImaginary = widget->getImag();
            zeros[i].multiplicity = widget->getMultiplicity();

            bool found = false;
            for (int prev = 0; prev < i; ++prev)
                if (doubleEquals(zeros[prev].real, zeros[i].real) &&
                    doubleEquals(zeros[prev].absImaginary, zeros[i].absImaginary))
                {
                    _zerosValidationOk = false;
                    found = true;
                    break;
                }

            widget->setValidationVisibility(found);
        }

        changesWereMade();
        propagateValidationResult();
    }

    void SetupECSpacesDialog::changesWereMade()
    {
        _wereAnyChangesMade =
            _editedSpace->definitionDomain != _originalSpace->definitionDomain ||
            _editedSpace->characteristicPolynomial.zeros != _originalSpace->characteristicPolynomial.zeros;
    }

    void SetupECSpacesDialog::propagateValidationResult()
    {
        _allValidationOk = _domainValidationOk && _zerosValidationOk;

        ui->saveButton->setToolTip("");
        ui->saveForLaterUseButton->setToolTip("");
        ui->saveButton->setEnabled(_wereAnyChangesMade && _allValidationOk);
        ui->saveForLaterUseButton->setEnabled(_allValidationOk);
        ui->undoChangesButton->setEnabled(_wereAnyChangesMade);

        if (!_wereAnyChangesMade)
            ui->saveButton->setToolTip("No changes to save.");

        if (!_allValidationOk) {
            ui->saveButton->setToolTip("Invalid state cannot be saved.");
            ui->saveForLaterUseButton->setToolTip("Invalid state cannot be saved.");
        }
    }

    void SetupECSpacesDialog::undoChangesPressed()
    {
        delete _editedSpace;
        _editedSpace = new ECSpace(*_originalSpace);
        loadEditorData();
    }

    void SetupECSpacesDialog::saveForLaterUsePressed()
    {
        auto dialog = new SaveForLaterUseDialog(this, _database, _editedSpace);
        dialog->exec();
        dialog->setParent(nullptr);
        delete dialog;
    }

    void SetupECSpacesDialog::loadExistingSetupPressed()
    {
        auto dialog = new LoadExistingSpaceDialog(this, _database, _editedSpace);
        if (dialog->exec())
        {
            loadEditorData();
        }
    }

    void SetupECSpacesDialog::savePressed()
    {
        *_originalSpace = *_editedSpace;
        _originalSpace->preprocessing();
        accept();
    }

    void SetupECSpacesDialog::reject()
    {
        if (!_wereAnyChangesMade) {
            QDialog::reject();
            return;
        }

        auto reply = QMessageBox::warning(
            this,
            "Cancel?",
            "Unsaved changes will be discarded. Continue?",
            QMessageBox::Yes|QMessageBox::No);

        if(reply == QMessageBox::Yes)
            QDialog::reject();
    }
}
