#include "PreviewECWidget.h"
#include "ui_PreviewECWidget.h"

#include <QColorDialog>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QScrollArea>
#include <QLabel>
#include <QGroupBox>
#include <sstream>
#include <iomanip>
#include "Utils/DoubleEquality.h"
#include "NNBBasesGLWidget.h"

namespace cagd
{
    PreviewECWidget::PreviewECWidget(QWidget *parent, ECSpace *space):
        QWidget(parent),
        ui(new Ui::PreviewECWidget),
        _space(space),
        _plotBackground(0,0,0)
    {
        _space->preprocessing();

        ui->setupUi(this);
        ui->scrollArea->setWidget(desctiptionWidget());

        auto plotter = new NNBBasesGLWidget(_space, this);
        ui->nnbLayout->addWidget(plotter);

        connect(this, SIGNAL(backgroundChanged(QColor)), plotter, SLOT(backgroundChanged(QColor)));

        connect(ui->colorButton, SIGNAL(clicked()), this, SLOT(colorButtonClicked()));
    }

    PreviewECWidget::~PreviewECWidget()
    {
        delete ui;
    }

    QWidget *PreviewECWidget::desctiptionWidget()
    {
        auto defDomain = new QLabel(this);
        defDomain->setText("Definition Domain:");
        defDomain->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
        auto diffEq = new QLabel(this);
        diffEq->setText("Differential Equation:");
        diffEq->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
        auto ordBasis = new QLabel(this);
        ordBasis->setText("Ordinary Basis Functions:");
        ordBasis->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

        auto leftLayout = new QVBoxLayout();
        leftLayout->addWidget(defDomain);
        leftLayout->addWidget(diffEq);
        leftLayout->addWidget(ordBasis);

        QLabel *label1 = new QLabel(this);
        label1->setTextFormat(Qt::TextFormat::RichText);
        label1->setText(domainRichText().c_str());
        label1->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Preferred);

        QLabel *label2 = new QLabel(this);
        label2->setTextFormat(Qt::TextFormat::RichText);
        label2->setText(diffEquationRichText().c_str());
        label2->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Preferred);

        QLabel *label3 = new QLabel(this);
        label3->setTextFormat(Qt::TextFormat::RichText);
        label3->setText(ordinaryBasesRichText().c_str());
        label3->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Preferred);

        auto rightLayout = new QVBoxLayout();
        rightLayout->addWidget(label1);
        rightLayout->addWidget(label2);
        rightLayout->addWidget(label3);

        auto layout = new QHBoxLayout();
        layout->addLayout(leftLayout);
        layout->addLayout(rightLayout);

        QWidget *widget = new QWidget();
        widget->setLayout(layout);
        return widget;
    }

    std::string PreviewECWidget::domainRichText()
    {
        return
            "<span style=\"font-size:14pt;\">[</span><span style=\"font-size:11pt;\"> " +
            format(_space->definitionDomain.first) +
            "</span><span style=\"font-size:12pt;\">,</span><span style=\"font-size:11pt;\"> " +
            format(_space->definitionDomain.second) +
                " </span><span style=\"font-size:14pt;\">]</span>";
    }

    std::string PreviewECWidget::diffEquationRichText()
    {
        std::stringstream ss;
        ss << "<p style=\"font-size:12pt;\">";

        auto coefficients = _space->characteristicPolynomial.getCoefficients();
        bool first = true;

        for (int order = (int) coefficients.size() - 1; order >= 0; order--) {
            double current = coefficients[order];
            if(doubleEquals(current, 0))
                continue;

            if(!first) {
                if(current < 0) {
                    current = -current;
                    ss << " - ";
                }
                else
                    ss << " + ";
            }

            if (!doubleEquals(std::abs(current), 1))
                ss << formatTrimmed(current) << "·";

            ss << "y<span style=\"vertical-align:super;\">("
               << order << ")</span>";
            first = false;
        }

        ss << " = 0</p>";
        return ss.str();
    }

    std::string PreviewECWidget::ordinaryBasesRichText()
    {
        auto separator = ", &nbsp;";

        std::stringstream ss;
        ss << "<p style=\"font-size:12pt;\">";

        bool first = true;
        for (auto base : _space->ordBases) {
            if (!first)
                ss << separator;
            first = false;

            ss << base.getHTMLRepresentation();
        }

        ss << "</p>";
        return ss.str();
    }

    std::string PreviewECWidget::format(double value)
    {
        std::stringstream ss;
        ss << std::fixed << std::setprecision(6) << value;
        return ss.str();
    }

    std::string PreviewECWidget::formatTrimmed(double value)
    {
        std::string result = format(value);

        int nrTrailingChars = 0;
        for (int i = (int) result.size() - 1; i > 0; --i)
            if (result[i] == '0')
                ++nrTrailingChars;
            else if (result[i] == '.') {
                ++nrTrailingChars;
                break;
            }
            else break;

        return result.substr(0, result.size() - nrTrailingChars);
    }

    void PreviewECWidget::colorButtonClicked()
    {
        auto result = QColorDialog::getColor(_plotBackground);
        if (result.isValid()) {
            _plotBackground = result;
            backgroundChanged(result);
        }
    }
}
