#include "NNBBasesGLWidget.h"
#include <iostream>
#include <cmath>
#include <cstdlib>
#include <ctime>
#include <GL/glu.h>
#include <QMessageBox>
#include <cstdlib>

#include "Core/Constants.h"
#include "Core/Exceptions.h"
#include "Core/Matrices.h"
#include "Core/Materials.h"

namespace cagd
{
    NNBBasesGLWidget::NNBBasesGLWidget(ECSpace *space, QWidget *parent, const QGLFormat &format):
        QGLWidget(format, parent),
        _space(space),
        _curves(space->getDimension(), nullptr),
        _backgroundColor {0, 0, 0}
    {}

    NNBBasesGLWidget::~NNBBasesGLWidget()
    {
        for (auto ptr : _curves)
            if(ptr) {
                ptr->DeleteVertexBufferObjects();
                delete ptr;
            }
    }

    void NNBBasesGLWidget::initializeGL()
    {
        double xMidPoint = (_space->definitionDomain.first + _space->definitionDomain.second) / 2;
        double horizScaleFactor = 3.0 / (_space->definitionDomain.second - _space->definitionDomain.first);

        glMatrixMode(GL_PROJECTION);
        glLoadIdentity();

        _aspect = (float)width() / (float)height();
        _z_near = 1.0f;
        _z_far = 1000.0f;
        _fovy = 45.0f;

        gluPerspective(_fovy, _aspect, _z_near, _z_far);

        glMatrixMode(GL_MODELVIEW);
        glLoadIdentity();

        _eye[0] = 0.0f; _eye[1] = 0.5f, _eye[2] = 1.4f;
        _center[0] = 0.0f; _center[1] = 0.5f; _center[2] = 0.0f;
        _up[0] = _up[2] = 0.0f, _up[1] = 1.0f;

        gluLookAt(_eye[0], _eye[1], _eye[2],
                  _center[0], _center[1], _center[2],
                  _up[0], _up[1], _up[2]);

        try {
            GLenum error = glewInit();

            if (error != GLEW_OK) {
                throw Exception("Could not initialize the OpenGL Extension Wrangler Library!");
            }

            if(!glewIsSupported("GL_VERSION_2_0")){
                throw Exception("Your graphics card is not compatible with OpenGL 2.0+! "
                                "Try to update your driver or buy a new graphics adapter!");
            }

            unsigned dimension = _space->getDimension();
            double divisionSize =
                (_space->definitionDomain.second - _space->definitionDomain.first) / (POINT_COUNT - 1);

            for (unsigned curveInd = 0; curveInd < dimension; ++curveInd) {
                _curves[curveInd] = new GenericCurve3(0, POINT_COUNT);

                for (unsigned pointInd = 0; pointInd < POINT_COUNT; ++pointInd) {
                    double paramValue = std::min(
                        _space->definitionDomain.second,
                        _space->definitionDomain.first + pointInd * divisionSize);
                    _curves[curveInd]->SetDerivative(
                        0,
                        pointInd,
                        DCoordinate3(
                            (paramValue - xMidPoint) * horizScaleFactor,
                            _space->NNBBaseDerivative(curveInd, 0, paramValue),
                            0
                        ));
                }

                _curves[curveInd]->UpdateVertexBufferObjects();
            }
        }
        catch (Exception &e) {
            QMessageBox::critical(nullptr, "Error", e.getReason().c_str());
            std::abort();
        }
    }


    void NNBBasesGLWidget::paintGL()
    {
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

        glPushMatrix();

            glClearColor(_backgroundColor[0], _backgroundColor[1], _backgroundColor[2], 1.0f);

            unsigned index = 0;
            for (auto curve : _curves)
            {
                if (!curve) continue;

                unsigned colorIndex = index < (_space->getDimension() + 1) / 2
                    ? index
                    : _space->getDimension() - 1 - index;

                setColor(colorIndex);
                glLineWidth(2);
                curve->RenderDerivatives(0, GL_LINE_STRIP);

                ++index;
            }

        glPopMatrix();
    }


    void NNBBasesGLWidget::resizeGL(int w, int h)
    {
        glViewport(0,0,w,h);

        glMatrixMode(GL_PROJECTION);
        glLoadIdentity();

        _aspect = (float) w / (float) h;
        gluPerspective(_fovy, _aspect, _z_near, _z_far);

        glMatrixMode(GL_MODELVIEW);
        updateGL();
    }

    void NNBBasesGLWidget::setColor(unsigned colorIndex)
    {
        static const float offset = 0.1f;
        static const float intensity = 1 - offset;

        static const float colors[][3] = {
            {1.0f, 0.0f, 0.0f},
            {0.0f, 0.0f, 1.0f},
            {0.0f, 1.0f, 0.0f},
            {0.8f, 0.0f, 1.0f},
            {0.75f, 0.75f, 0.0f}
        };

        colorIndex %= (unsigned) (sizeof(colors) / sizeof(colors[0]));

        glColor3f(
            offset + intensity * colors[colorIndex][0],
            offset + intensity * colors[colorIndex][1],
                offset + intensity * colors[colorIndex][2]);
    }

    void NNBBasesGLWidget::backgroundChanged(QColor value)
    {
        qreal r,g,b;
        value.getRgbF(&r, &g, &b);
        _backgroundColor[0] = (float) r;
        _backgroundColor[1] = (float) g;
        _backgroundColor[2] = (float) b;
        repaint();
    }
}

