#include "SurfaceItem.h"
#include "../CommonUITools.h"
#include "../../Core/Lights.h"
#include "../../Core/GeometryHelper.h"

namespace cagd
{
    SurfaceItem::SurfaceItem(Scene *scene):
        SceneItem(scene),
        ecSpaceU(nullptr),
        ecSpaceV(nullptr),
        bsurface(nullptr),
        image(nullptr),
        uIsoLines(nullptr),
        vIsoLines(nullptr)
    {
        material = CommonUITools::getRandomMaterial();
        normalsColor = CommonUITools::getRandomLightColor();
        isoLinesColor = CommonUITools::getRandomLightColor();
        alwaysShowControlNet = false;
        showIsoparametricLines = false;
        showNormals = false;
    }

    SurfaceItem::~SurfaceItem()
    {
        if (ecSpaceU != nullptr) delete ecSpaceU;
        if (ecSpaceV != nullptr) delete ecSpaceV;
        if (bsurface != nullptr) delete bsurface;
        if (image != nullptr) delete image;
        deleteIsoLines();
    }

    void SurfaceItem::render(bool isSelected)
    {
        // surface
        glEnable(GL_LIGHTING);
        scene->shaderProgram->Enable();
        if (image) {
            material->Apply();
            image->Render(GL_TRIANGLES);
        }
        scene->shaderProgram->Disable();
        glDisable(GL_LIGHTING);

        // control nets
        if (isSelected || alwaysShowControlNet) {
            CommonUITools::setRenderingColor(scene->controlLinesColor);
            glLineWidth(isSelected ? scene->lineWidthForSelection : scene->lineWidth);
            bsurface->RenderData(GL_LINE_STRIP);
        }

        // normal vectors
        if (showNormals) {
            glLineWidth(scene->lineWidth);
            CommonUITools::setRenderingColor(normalsColor);
            if (image)
                image->RenderNormals();
        }

        // isoparametric lines
        if (showIsoparametricLines) {
            glLineWidth(scene->lineWidth);
            CommonUITools::setRenderingColor(isoLinesColor);

            if (uIsoLines)
                for(unsigned i = 0; i < uIsoLines->columnCount(); ++i)
                    (*uIsoLines)[i]->RenderDerivatives(0, GL_LINE_STRIP);
            if (vIsoLines)
                for(unsigned i = 0; i < vIsoLines->columnCount(); ++i)
                    (*vIsoLines)[i]->RenderDerivatives(0, GL_LINE_STRIP);
        }
    }

    void SurfaceItem::update()
    {
        bsurface->UpdateVertexBufferObjectsOfData();
        if (image) delete image;
        deleteIsoLines();

        image = bsurface->GenerateImage(scene->surfaceDivPointCount, scene->surfaceDivPointCount);
        image->UpdateVertexBufferObjects();

        uIsoLines = bsurface->GenerateUIsoparametricLines(
            scene->isoparametricLineCount,
            0,
            scene->surfaceDivPointCount);

        for(unsigned i = 0; i < uIsoLines->columnCount(); ++i)
            (*uIsoLines)[i]->UpdateVertexBufferObjects();

        vIsoLines = bsurface->GenerateVIsoparametricLines(
            scene->isoparametricLineCount,
            0,
            scene->surfaceDivPointCount);

        for(unsigned i = 0; i < vIsoLines->columnCount(); ++i)
            (*vIsoLines)[i]->UpdateVertexBufferObjects();
    }

    void SurfaceItem::regenerate()
    {
        auto oldSurface = bsurface;

        bsurface = new BSurface3(ecSpaceU, ecSpaceV);

        for (unsigned i = 0; i < ecSpaceU->getDimension() && i < oldSurface->GetUControlPointCount(); ++i)
            for (unsigned j = 0; j < ecSpaceV->getDimension() && j < oldSurface->GetVControlPointCount(); ++j)
                (*bsurface)(i,j) = (*oldSurface)(i,j);

        for (unsigned i = oldSurface->GetUControlPointCount(); i < ecSpaceU->getDimension(); ++i)
            for (unsigned j = 0; j < ecSpaceV->getDimension(); ++j)
                if (i >= 1 && j>=1)
                    (*bsurface)(i,j) = (*bsurface)(i-1, j) + (*bsurface)(i, j-1) - (*bsurface)(i-1, j-1);
                else if (i >= 2)
                    (*bsurface)(i,j) = 2 * (*bsurface)(i-1, j) - (*bsurface)(i-2, j);
                else if (j >= 2)
                    (*bsurface)(i,j) = 2 * (*bsurface)(i, j-1) - (*bsurface)(i, j-2);
                else if (i >= 1)
                    (*bsurface)(i,j) = (*bsurface)(i-1, j) + DCoordinate3(
                        1.0 * std::rand() / RAND_MAX,
                        1.0 * std::rand() / RAND_MAX,
                        1.0 * std::rand() / RAND_MAX);
                else if (j >= 1)
                    (*bsurface)(i,j) = (*bsurface)(i, j-1) + DCoordinate3(
                        1.0 * std::rand() / RAND_MAX,
                        1.0 * std::rand() / RAND_MAX,
                        1.0 * std::rand() / RAND_MAX);
                else
                    (*bsurface)(i,j) = DCoordinate3(0,0,0);

        for (unsigned i = 0; i < ecSpaceU->getDimension(); ++i)
            for (unsigned j = oldSurface->GetVControlPointCount(); j < ecSpaceV->getDimension(); ++j)
                if (i >= 1 && j>=1)
                    (*bsurface)(i,j) = (*bsurface)(i-1, j) + (*bsurface)(i, j-1) - (*bsurface)(i-1, j-1);
                else if (i >= 2)
                    (*bsurface)(i,j) = 2 * (*bsurface)(i-1, j) - (*bsurface)(i-2, j);
                else if (j >= 2)
                    (*bsurface)(i,j) = 2 * (*bsurface)(i, j-1) - (*bsurface)(i, j-2);
                else if (i >= 1)
                    (*bsurface)(i,j) = (*bsurface)(i-1, j) + DCoordinate3(
                        1.0 * std::rand() / RAND_MAX,
                        1.0 * std::rand() / RAND_MAX,
                        1.0 * std::rand() / RAND_MAX);
                else if (j >= 1)
                    (*bsurface)(i,j) = (*bsurface)(i, j-1) + DCoordinate3(
                        1.0 * std::rand() / RAND_MAX,
                        1.0 * std::rand() / RAND_MAX,
                        1.0 * std::rand() / RAND_MAX);
                else
                    (*bsurface)(i,j) = DCoordinate3(0,0,0);

        delete oldSurface;
        update();
    }

    void SurfaceItem::deleteIsoLines()
    {
        if (uIsoLines != nullptr) {
            for(unsigned i = 0; i < uIsoLines->columnCount(); ++i)
                if ((*uIsoLines)[i] != nullptr) delete (*uIsoLines)[i];

            delete uIsoLines;
            uIsoLines = nullptr;
        }

        if (vIsoLines != nullptr) {
            for(unsigned i = 0; i < vIsoLines->columnCount(); ++i)
                if ((*vIsoLines)[i] != nullptr) delete (*vIsoLines)[i];

            delete vIsoLines;
            vIsoLines = nullptr;
        }
    }

    SceneItem::PickResult SurfaceItem::pickControlPointClosestToRay(HCoordinate3 rayPoint1, HCoordinate3 rayPoint2)
    {
        PickResult result;
        result.distance = std::numeric_limits<double>::infinity();

        for (unsigned i = 0; i < bsurface->GetUControlPointCount(); ++i)
            for (unsigned j = 0; j < bsurface->GetVControlPointCount(); ++j) {
                double d = GeometryHelper::distanceFromLine((*bsurface)(i,j), rayPoint1, rayPoint2);

                if (d < result.distance) {
                    result.distance = d;
                    result.point = &(*bsurface)(i,j);
                }
            }

        return result;
    }

    double SurfaceItem::itemDistanceFromPointIfCloseToRay(
        HCoordinate3 point,
        HCoordinate3 otherRayPoint,
        double rayDistanceThreshold)
    {
        double distance = std::numeric_limits<double>::infinity();

        for (unsigned i = 0; i < image->VertexCount(); ++i) {
            double rayDistance = GeometryHelper::distanceFromLine(image->GetVertex(i), point, otherRayPoint);
            if (rayDistance <= rayDistanceThreshold)
                distance = std::min(distance, (image->GetVertex(i) - point).length());
        }

        return distance;
    }

    void SurfaceItem::transform(const RealMatrix &transformation)
    {
        for (unsigned i = 0; i < bsurface->GetUControlPointCount(); ++i)
            for (unsigned j = 0; j < bsurface->GetVControlPointCount(); ++j)
                (*bsurface)(i,j) = GeometryHelper::transformPoint((*bsurface)(i,j), transformation);
        bsurface->UpdateVertexBufferObjectsOfData();

        image->transform(transformation);

        if (uIsoLines)
            for(unsigned i = 0; i < uIsoLines->columnCount(); ++i)
                (*uIsoLines)[i]->transform(transformation);
        if (vIsoLines)
            for(unsigned i = 0; i < vIsoLines->columnCount(); ++i)
                (*vIsoLines)[i]->transform(transformation);
    }

    DCoordinate3 SurfaceItem::centerOfGravity()
    {
        DCoordinate3 result(0, 0, 0);

        for (unsigned i = 0; i < bsurface->GetUControlPointCount(); ++i)
            for (unsigned j = 0; j < bsurface->GetVControlPointCount(); ++j)
                result += (*bsurface)(i,j);

        return result / (bsurface->GetUControlPointCount() * bsurface->GetVControlPointCount());
    }

    void SurfaceItem::performOrderElevationU(double reZero, double imZero, int multiplicity)
    {
        auto result = bsurface->performUOrderElevation(reZero, imZero, multiplicity);

        if (result.first == nullptr)
            return;

        delete ecSpaceU;
        delete bsurface;

        ecSpaceU = result.first;
        bsurface = result.second;

        update();
    }

    void SurfaceItem::performOrderElevationV(double reZero, double imZero, int multiplicity)
    {
        auto result = bsurface->performVOrderElevation(reZero, imZero, multiplicity);

        if (result.first == nullptr)
            return;

        delete ecSpaceV;
        delete bsurface;

        ecSpaceV = result.first;
        bsurface = result.second;

        update();
    }

    void SurfaceItem::performSubdivisionU(double point)
    {
        auto result = bsurface->performUSubdivision(point);

        if (result.bSurfaceLeft == nullptr)
            return;

        delete ecSpaceU;
        delete bsurface;

        ecSpaceU = result.ecSpaceLeft;
        bsurface = result.bSurfaceLeft;
        update();

        auto other = new SurfaceItem(scene);
        other->ecSpaceV = new ECSpace(*result.bSurfaceRight->getECSpaceV());
        other->ecSpaceV->preprocessing();
        other->ecSpaceU = result.ecSpaceRight;
        other->bsurface = result.bSurfaceRight;
        scene->items.push_back(other);
        other->update();
    }

    void SurfaceItem::performSubdivisionV(double point)
    {
        auto result = bsurface->performVSubdivision(point);

        if (result.bSurfaceLeft == nullptr)
            return;

        delete ecSpaceV;
        delete bsurface;

        ecSpaceV = result.ecSpaceLeft;
        bsurface = result.bSurfaceLeft;
        update();

        auto other = new SurfaceItem(scene);
        other->ecSpaceU = new ECSpace(*result.bSurfaceRight->getECSpaceU());
        other->ecSpaceU->preprocessing();
        other->ecSpaceV = result.ecSpaceRight;
        other->bsurface = result.bSurfaceRight;
        scene->items.push_back(other);
        other->update();
    }

    void SurfaceItem::updateControlPointsForExactDescription(const std::vector<DCoordinate3> &coefficients)
    {
        bsurface->updateControlPointsForExactDescription(coefficients);
        update();
    }

    std::ostream &operator<<(std::ostream &out, const SurfaceItem &surfaceItem)
    {
        out << *surfaceItem.ecSpaceU << std::endl;
        out << *surfaceItem.ecSpaceV << std::endl;

        for (unsigned i = 0; i < surfaceItem.bsurface->GetUControlPointCount(); ++i)
            for (unsigned j = 0; j < surfaceItem.bsurface->GetVControlPointCount(); ++j)
                out << (*surfaceItem.bsurface)(i,j) << std::endl;

        auto &materials = CommonUITools::listOfMaterials;
        out << std::find(materials.begin(), materials.end(), surfaceItem.material) - materials.begin() << std::endl;

        out << surfaceItem.normalsColor << std::endl;
        out << surfaceItem.isoLinesColor << std::endl;
        out << surfaceItem.alwaysShowControlNet << std::endl;
        out << surfaceItem.showIsoparametricLines << std::endl;
        out << surfaceItem.showNormals << std::endl;

        return out;
    }

    std::istream &operator>>(std::istream &in, SurfaceItem &surfaceItem)
    {
        if (surfaceItem.ecSpaceU) delete surfaceItem.ecSpaceU;
        if (surfaceItem.ecSpaceV) delete surfaceItem.ecSpaceV;
        if (surfaceItem.bsurface) delete surfaceItem.bsurface;

        surfaceItem.ecSpaceU = new ECSpace();
        surfaceItem.ecSpaceV = new ECSpace();
        in >> *surfaceItem.ecSpaceU;
        in >> *surfaceItem.ecSpaceV;
        surfaceItem.ecSpaceU->preprocessing();
        surfaceItem.ecSpaceV->preprocessing();

        surfaceItem.bsurface = new BSurface3(surfaceItem.ecSpaceU, surfaceItem.ecSpaceV);

        for (unsigned i = 0; i < surfaceItem.bsurface->GetUControlPointCount(); ++i)
            for (unsigned j = 0; j < surfaceItem.bsurface->GetVControlPointCount(); ++j)
                in >> (*surfaceItem.bsurface)(i,j);

        int materialIndex;
        in >> materialIndex;
        surfaceItem.material = CommonUITools::listOfMaterials[materialIndex];

        in >> surfaceItem.normalsColor;
        in >> surfaceItem.isoLinesColor;
        in >> surfaceItem.alwaysShowControlNet;
        in >> surfaceItem.showIsoparametricLines;
        in >> surfaceItem.showNormals;

        surfaceItem.update();
        return in;
    }
}
