#include "CurveItem.h"
#include "../CommonUITools.h"
#include <limits>
#include "../../Core/GeometryHelper.h"

namespace cagd
{
    CurveItem::CurveItem(Scene *scene):
        SceneItem(scene),
        ecSpace(nullptr),
        bcurve(nullptr),
        image(nullptr)
    {
        color = CommonUITools::getRandomLightColor();
        derivColor = CommonUITools::getRandomLightColor();
        secondDerivColor = CommonUITools::getRandomLightColor();

        alwaysShowControlPolygon = false;
        showFirstDerivatives = false;
        showSecondDerivatives = false;
    }

    CurveItem::~CurveItem()
    {
        if (ecSpace) delete ecSpace;
        if (bcurve) delete bcurve;
        if (image) delete image;
    }

    void CurveItem::render(bool isSelected)
    {
        // curve
        glLineWidth(isSelected ? scene->lineWidthForSelection : scene->lineWidth);
        CommonUITools::setRenderingColor(color);
        if (image)
            image->RenderDerivatives(0, GL_LINE_STRIP);

        // first order derivatives
        if (showFirstDerivatives) {
            CommonUITools::setRenderingColor(derivColor);
            if (image)
                image->RenderDerivatives(1, GL_LINES);
        }

        // second order derivatives
        if (showSecondDerivatives) {
            CommonUITools::setRenderingColor(secondDerivColor);
            if (image)
                image->RenderDerivatives(2, GL_LINES);
        }

        // control polygon
        if (isSelected || alwaysShowControlPolygon) {
           CommonUITools::setRenderingColor(scene->controlLinesColor);
           glLineWidth(isSelected ? scene->lineWidthForSelection : scene->lineWidth);
           if(bcurve)
               bcurve->RenderData(GL_LINE_STRIP);
        }
    }

    void CurveItem::update()
    {
        bcurve->UpdateVertexBufferObjectsOfData();
        if (image) delete image;
        image = bcurve->GenerateImage(scene->curveMaxDerivOrder, scene->curveDivPointCount);
        image->UpdateVertexBufferObjects();
    }

    void CurveItem::regenerate()
    {
        auto oldCurve = bcurve;

        bcurve = new BCurve3(ecSpace);

        for (unsigned i = 0; i < ecSpace->getDimension() && i < oldCurve->GetControlPointCount(); ++i)
            (*bcurve)[i] = (*oldCurve)[i];

        for (unsigned i = oldCurve->GetControlPointCount(); i < ecSpace->getDimension(); ++i)
            if (i >= 1)
                (*bcurve)[i] = (*bcurve)[i-1] + DCoordinate3(
                    1.0 * std::rand() / RAND_MAX,
                    1.0 * std::rand() / RAND_MAX,
                    1.0 * std::rand() / RAND_MAX);
            else
                (*bcurve)[i] = DCoordinate3(0,0,0);

        delete oldCurve;
        update();
    }

    SceneItem::PickResult CurveItem::pickControlPointClosestToRay(HCoordinate3 rayPoint1, HCoordinate3 rayPoint2)
    {
        PickResult result;
        result.distance = std::numeric_limits<double>::infinity();

        for (unsigned i = 0; i < bcurve->GetControlPointCount(); ++i) {
            double d = GeometryHelper::distanceFromLine((*bcurve)[i], rayPoint1, rayPoint2);

            if (d < result.distance) {
                result.distance = d;
                result.point = &(*bcurve)[i];
            }
        }

        return result;
    }

    double CurveItem::itemDistanceFromPointIfCloseToRay(
        HCoordinate3 point,
        HCoordinate3 otherRayPoint,
        double rayDistanceThreshold)
    {
        double distance = std::numeric_limits<double>::infinity();

        for (unsigned i = 0; i < image->GetPointCount(); ++i) {
            double rayDistance = GeometryHelper::distanceFromLine((*image)(0, i), point, otherRayPoint);
            if (rayDistance <= rayDistanceThreshold)
                distance = std::min(distance, ((*image)(0,i) - point).length());
        }

        return distance;
    }

    void CurveItem::transform(const RealMatrix &transformation)
    {
        for (unsigned i = 0; i < bcurve->GetControlPointCount(); ++i)
            (*bcurve)[i] = GeometryHelper::transformPoint((*bcurve)[i], transformation);
        bcurve->UpdateVertexBufferObjectsOfData();

        image->transform(transformation);
    }

    DCoordinate3 CurveItem::centerOfGravity()
    {
        DCoordinate3 result(0, 0, 0);
        for (unsigned i = 0; i < bcurve->GetControlPointCount(); ++i)
            result += (*bcurve)[i];
        return result / bcurve->GetControlPointCount();
    }

    void CurveItem::performOrderElevation(double real, double absImag, unsigned multiplicity)
    {
        auto result = bcurve->performOrderElevation(real, absImag, multiplicity);

        if (result.first == nullptr)
            return;

        delete ecSpace;
        delete bcurve;
        ecSpace = result.first;
        bcurve = result.second;

        update();
    }

    void CurveItem::performSubdivision(double gamma)
    {
        auto result = bcurve->performSubdivision(gamma);

        if (result.bCurveLeft == nullptr)
            return;

        delete ecSpace;
        delete bcurve;
        ecSpace = result.ecSpaceLeft;
        bcurve = result.bCurveLeft;
        update();

        auto other = new CurveItem(scene);
        other->ecSpace = result.ecSpaceRight;
        other->bcurve = result.bCurveRight;
        scene->items.push_back(other);
        other->update();
    }

    void CurveItem::updateControlPointsForExactDescription(const std::vector<DCoordinate3> &coefficients)
    {
        bcurve->updateControlPointsForExactDescription(coefficients);
        update();
    }

    std::ostream &operator<<(std::ostream &out, const CurveItem &curveItem)
    {
        out << *curveItem.ecSpace << std::endl;

        for (unsigned i = 0; i < curveItem.bcurve->GetControlPointCount(); ++i)
            out << (*curveItem.bcurve)[i] << std::endl;

        out << curveItem.color << std::endl;
        out << curveItem.derivColor << std::endl;
        out << curveItem.secondDerivColor << std::endl;
        out << curveItem.alwaysShowControlPolygon << std::endl;
        out << curveItem.showFirstDerivatives << std::endl;
        out << curveItem.showSecondDerivatives << std::endl;
        return out;
    }

    std::istream &operator>>(std::istream &in, CurveItem &curveItem)
    {
        if (curveItem.ecSpace) delete curveItem.ecSpace;
        if (curveItem.bcurve) delete curveItem.bcurve;

        curveItem.ecSpace = new ECSpace();
        in >> *curveItem.ecSpace;
        curveItem.ecSpace->preprocessing();

        curveItem.bcurve = new BCurve3(curveItem.ecSpace);
        for (unsigned i = 0; i < curveItem.bcurve->GetControlPointCount(); ++i)
            in >> (*curveItem.bcurve)[i];

        in >> curveItem.color;
        in >> curveItem.derivColor;
        in >> curveItem.secondDerivColor;
        in >> curveItem.alwaysShowControlPolygon;
        in >> curveItem.showFirstDerivatives;
        in >> curveItem.showSecondDerivatives;

        curveItem.update();
        return in;
    }
}
