#pragma once

#include <GL/glew.h>
#include <QGLWidget>
#include <QGLFormat>
#include <QTimer>
#include <QTemporaryDir>

#include "EC/BCurve3.h"
#include "Core/GenericCurves3.h"
#include "./SetupECSpaces/ECSpaceDatabase.h"
#include "Model/Scene.h"
#include "../Core/Lights.h"
#include "../Core/ShaderPrograms.h"
#include "../Core/TriangulatedMeshes3.h"


namespace cagd
{
    class GLWidget: public QGLWidget
    {
        Q_OBJECT

    private:
        enum TransformType {
            None,
            SceneTranslation,
            SceneRotation,
            ItemTranslation,
            ItemRotation,
            PointTranslation
        };

        static const int MOUSE_THRESHOLD = 5;

        // variables defining the projection matrix
        float       _aspect;            // aspect ratio of the rendering window
        float       _fovy;              // field of view in direction y
        float       _z_near, _z_far;    // distance of near and far clipping planes

        // variables defining the model-view matrix
        float       _eye[3], _center[3], _up[3];

        int _widgetWidth, _widgetHeight;
        PointLight *_light = nullptr;

        Scene *_scene;

        RealMatrix _sceneTransformation;
        bool _isInverseActualized;
        RealMatrix _sceneTransformationInverse;

        int _startPosW;
        int _startPosH;
        int _prevPosW;
        int _prevPosH;

        bool _isLeftClick = false;
        bool _isRightClick = false;

        Scene::Selection highlight;

        TransformType _currentTransform;
        RealMatrix _additionalSceneTransformation;
        DCoordinate3 _prevTransAdded;
        RealMatrix _itemRotationState;
        DCoordinate3 _itemRotationCenterOfGravity;

        QTemporaryDir _temporaryDir;
        ShaderProgram _shader;

        TriangulatedMesh3 *_pointIndicator;


    public:
        GLWidget(QWidget *parent, Scene *scene,
                 const QGLFormat &format = QGL::Rgba | QGL::DepthBuffer | QGL::DoubleBuffer);

        void initializeGL();
        void paintGL();
        void resizeGL(int w, int h);

        virtual ~GLWidget();

        virtual void mousePressEvent(QMouseEvent *);
        virtual void mouseReleaseEvent(QMouseEvent *);
        virtual void mouseMoveEvent(QMouseEvent *);
        virtual void wheelEvent(QWheelEvent *);
        virtual void keyPressEvent(QKeyEvent *);

    signals:
        void initialized();
        void modifiedTheSelection();

    public slots:
        void setAsCurrent();
        void clearViewTransformations();
        void moveSelectionToCenter();

    private:
        void updateTransformInverse();
        void tryToPickElement(int posW, int posH);
        void updateCursor();

        void transform(int posW, int posH);
        void commitTransformation(int posW, int posH);
        void cancelTransformation();

        void translatePoint(int posW, int posH);
        void translateItem(int posW, int posH);
        void rotateItem(int posW, int posH);
        void translateScene(int posW, int posH);
        void rotateScene(int posW, int posH);

        HCoordinate3 viewCoordFromScreen(int posW, int posH, double planeZ);
        HCoordinate3 modelCoordFromScreen(int posW, int posH, double planeZ);
    };
}
