#include "GLWidget.h"
#include <iostream>
#include <iomanip>
#include <cmath>
#include <cstdlib>
#include <ctime>
#include <GL/glu.h>
#include <QMouseEvent>
#include <QFile>
#include <QMessageBox>
#include <cstdlib>

#include "Core/RealMatrixDecompositions.h"
#include "Core/Constants.h"
#include "Core/Exceptions.h"
#include "Core/Matrices.h"
#include "Core/Materials.h"
#include "Core/GeometryHelper.h"

namespace cagd
{
    GLWidget::GLWidget(QWidget *parent, Scene *scene, const QGLFormat &format):
        QGLWidget(format, parent),
        _scene(scene),
        _sceneTransformation(4, 4),
        _sceneTransformationInverse(4, 4),
        _additionalSceneTransformation(4, 4),
        _itemRotationState(4,4)
    {
        setMouseTracking(true);
        setFocusPolicy(Qt::StrongFocus);

        _sceneTransformation.loadIdentityMatrix();
        _isInverseActualized = true;

        _sceneTransformationInverse.loadIdentityMatrix();
        _additionalSceneTransformation.loadIdentityMatrix();
        _prevTransAdded = DCoordinate3(0,0,0);
        _currentTransform = None;
        _itemRotationState.loadIdentityMatrix();
    }

    GLWidget::~GLWidget()
    {
        delete _light;
        if (_pointIndicator)
            delete _pointIndicator;
    }

    void GLWidget::initializeGL()
    {
        glMatrixMode(GL_PROJECTION);
        glLoadIdentity();

        _widgetWidth = width();
        _widgetHeight = height();

        _aspect = (float)width() / (float)height();
        _z_near = 1.0;
        _z_far = 1000.0;
        _fovy = 45.0;

        gluPerspective(_fovy, _aspect, _z_near, _z_far);

        glMatrixMode(GL_MODELVIEW);
        glLoadIdentity();

        _eye[0] = _eye[1] = 0.0, _eye[2] = 6.0;
        _center[0] = _center[1] = _center[2] = 0.0;
        _up[0] = _up[2] = 0.0, _up[1] = 1.0;

        gluLookAt(_eye[0], _eye[1], _eye[2],
                  _center[0], _center[1], _center[2],
                  _up[0], _up[1], _up[2]);

        try {
            GLenum error = glewInit();

            if (error != GLEW_OK) {
                throw Exception("Could not initialize the OpenGL Extension Wrangler Library!");
            }

            if(!glewIsSupported("GL_VERSION_2_0")){
                throw Exception("Your graphics card is not compatible with OpenGL 2.0+! "
                                "Try to update your driver or buy a new graphics adapter!");
            }       

            glEnable(GL_DEPTH_TEST);
            glEnable(GL_NORMALIZE);

            _light = new PointLight(
                GL_LIGHT0,
                HCoordinate3(0.0f, 0.0f, 12.0f, 1.0f),
                Color4(1.0f,1.0f,1.0f), Color4(0.6f,0.6f,0.6f), Color4(0.6f,0.6f,0.6f),
                0.01f, 0.08f, 0.0f);
            _light->Enable();

            glDisable(GL_LIGHTING);

            QFile::copy(":/Resources/Shaders/two_sided_lighting.vert",
                _temporaryDir.path() + "/two_sided_lighting.vert");
            QFile::copy(":/Resources/Shaders/two_sided_lighting.frag",
                _temporaryDir.path() + "/two_sided_lighting.frag");
            QFile::copy(":/Resources/sphere.off",
                _temporaryDir.path() + "/sphere.off");

            _shader.InstallShaders(
                (_temporaryDir.path() + "/two_sided_lighting.vert").toUtf8().constData(),
                (_temporaryDir.path() + "/two_sided_lighting.frag").toUtf8().constData()
            );
            _scene->shaderProgram = &_shader;

            _pointIndicator = new TriangulatedMesh3;
            _pointIndicator->LoadFromOFF((_temporaryDir.path() + "/sphere.off").toUtf8().constData());
            _pointIndicator->UpdateVertexBufferObjects();

            initialized();
        }
        catch (Exception &e) {
            QMessageBox::critical(nullptr, "Error", e.getReason().c_str());
            std::abort();
        }
    }

    void GLWidget::setAsCurrent()
    {
        makeCurrent();
    }

    void GLWidget::paintGL()
    {
        RealMatrix totalTransform = _additionalSceneTransformation * _sceneTransformation;

        GLfloat transform[16];
        for (unsigned i = 0; i < 4; ++i)
            for (unsigned j = 0; j < 4; ++j)
                transform[j*4 + i] = (GLfloat) totalTransform(i, j);

        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

        glClearColor(
            _scene->backgroundColor.r(),
            _scene->backgroundColor.g(),
            _scene->backgroundColor.b(),
            _scene->backgroundColor.a());

        glPushMatrix();

            glMultMatrixf(transform);

            _scene->render();

            if (_scene->selection.selectedPoint) {
                DCoordinate3 *point = _scene->selection.selectedPoint;

                glEnable(GL_LIGHTING);
                glPushMatrix();
                        glTranslatef((float) point->x(), (float) point->y(), (float) point->z());
                        glScalef(0.07f, 0.07f, 0.07f);
                        MatFBBrass.Apply();
                        _pointIndicator->Render();
                glPopMatrix();
                glDisable(GL_LIGHTING);
            }

        glPopMatrix();
    }

    void GLWidget::resizeGL(int w, int h)
    {
        _widgetWidth = w;
        _widgetHeight = h;

        glViewport(0,0,w,h);

        glMatrixMode(GL_PROJECTION);
        glLoadIdentity();

        _aspect = (float) w / (float) h;
        gluPerspective(_fovy, _aspect, _z_near, _z_far);

        glMatrixMode(GL_MODELVIEW);
        updateGL();
    }

    void GLWidget::mousePressEvent(QMouseEvent *event)
    {
        if (_isLeftClick || _isRightClick) {
            event->accept(); // do not consider any second button pressed
            return;
        }

        if (event->button() == Qt::LeftButton) {
            _isLeftClick = true;            
        }
        else if (event->button() == Qt::RightButton) {
            _isRightClick = true;
        }

        auto pos = event->pos();
        _startPosW = _prevPosW = pos.x();
        _startPosH = _prevPosH = pos.y();
        event->accept();
    }

    void GLWidget::mouseReleaseEvent(QMouseEvent *event)
    {
        auto pos = event->localPos();

        if (_currentTransform == None && _isLeftClick && _scene->selection != highlight) {
            _scene->selection = highlight;
            _scene->shouldTriggerTabChange = true;
            updateGL();
            modifiedTheSelection();
        }

        if (_currentTransform != None) {
            commitTransformation((int) pos.x(), (int) pos.y());
        }

        _isLeftClick = _isRightClick = false;

        tryToPickElement((int) pos.x(), (int) pos.y());
        updateCursor();
        event->accept();
    }

    void GLWidget::mouseMoveEvent(QMouseEvent *event)
    {
        event->accept();
        auto pos = event->localPos();

        if (std::abs(pos.x() - _prevPosW) < MOUSE_THRESHOLD && std::abs(pos.y() - _prevPosH) < MOUSE_THRESHOLD)
            return;

        if (_isLeftClick || _isRightClick) {
            transform((int) pos.x(), (int) pos.y());
            updateCursor();
        }
        else {
            tryToPickElement((int) pos.x(), (int) pos.y());
            updateCursor();
        }
    }

    void GLWidget::wheelEvent(QWheelEvent *event)
    {
        int units = event->delta() / 50;
        double factor = std::pow(0.95, units);

        if (highlight == _scene->selection && event->modifiers().testFlag(Qt::ControlModifier)) {
            DCoordinate3 targetPoint = _scene->selection.selectedItem->centerOfGravity();

            RealMatrix preTraslate = GeometryHelper::generateTranslationMatrix(-targetPoint);

            RealMatrix scaler(4, 4);
            for (unsigned i = 0; i < 4; ++i)
                for (unsigned j = 0; j < 4; ++j)
                    scaler(i, j) = (i == j);
            scaler(3, 3) = factor;

            RealMatrix postTranslate = GeometryHelper::generateTranslationMatrix(targetPoint);

            _scene->selection.selectedItem->transform(postTranslate * scaler * preTraslate);
            modifiedTheSelection();
        }
        else {
            DCoordinate3 targetPoint = viewCoordFromScreen(event->pos().x(), event->pos().y(), 0);

            RealMatrix preTraslate = GeometryHelper::generateTranslationMatrix(-targetPoint);

            RealMatrix scaler(4, 4);
            for (unsigned i = 0; i < 4; ++i)
                for (unsigned j = 0; j < 4; ++j)
                    scaler(i, j) = (i == j);
            scaler(3, 3) = factor;

            RealMatrix postTranslate = GeometryHelper::generateTranslationMatrix(targetPoint);

            _sceneTransformation = postTranslate * scaler * preTraslate * _sceneTransformation;
            _isInverseActualized = false;
        }

        updateGL();
        event->accept();
    }

    void GLWidget::keyPressEvent(QKeyEvent *event)
    {
        if (event->key() == Qt::Key_Escape)
            cancelTransformation();

        updateCursor();
        event->accept();
    }

    void GLWidget::transform(int posW, int posH)
    {
        if (_currentTransform == None) {
            bool decided = false;
            if (_scene->selection == highlight){
                if (_isLeftClick && highlight.isPoint()) {
                    _currentTransform = PointTranslation;
                    decided = true;
                }
                if (_isLeftClick && highlight.isWholeItem()) {
                    _currentTransform = ItemTranslation;
                    decided = true;
                }
                if (_isRightClick && highlight.isWholeItem()) {
                    _currentTransform = ItemRotation;
                    _itemRotationState.loadIdentityMatrix();
                    _itemRotationCenterOfGravity = GeometryHelper::transformPoint(
                        highlight.selectedItem->centerOfGravity(),
                        _sceneTransformation);
                    decided = true;
                }
            }

            if (!decided)
                _currentTransform = _isLeftClick ? SceneTranslation : SceneRotation;
        }

        switch (_currentTransform) {
            case PointTranslation:
                translatePoint(posW, posH);
                break;
            case ItemTranslation:
                translateItem(posW, posH);
                break;
            case ItemRotation:
                rotateItem(posW, posH);
                break;
            case SceneTranslation:
                translateScene(posW, posH);
                break;
            case SceneRotation:
                rotateScene(posW, posH);
                break;
            case None:
                break;
        }

        _prevPosW = posW;
        _prevPosH = posH;
    }

    void GLWidget::commitTransformation(int posW, int posH)
    {
        transform(posW, posH);

        switch (_currentTransform) {
            case SceneTranslation:
            case SceneRotation:
                _sceneTransformation = _additionalSceneTransformation * _sceneTransformation;
                _isInverseActualized = false;
                _additionalSceneTransformation.loadIdentityMatrix();
                updateGL();
                break;
            case ItemTranslation:
            case ItemRotation:
            case PointTranslation:
            case None:
                break;
        }

        _prevTransAdded = DCoordinate3(0, 0, 0);
        _isLeftClick = _isRightClick = false;
        _currentTransform = None;
    }

    void GLWidget::cancelTransformation()
    {
        auto point = highlight.selectedPoint;
        auto item = highlight.selectedItem;

        switch (_currentTransform) {
            case PointTranslation:
                *point -= _prevTransAdded;
                _prevTransAdded = DCoordinate3(0, 0, 0);

                item->update();
                updateGL();
                modifiedTheSelection();
                break;

            case ItemTranslation:
                item->transform(
                    GeometryHelper::generateTranslationMatrix(- _prevTransAdded));
                _prevTransAdded = DCoordinate3(0, 0, 0);

                updateGL();
                modifiedTheSelection();
                break;
            case ItemRotation:
                item->transform(
                    GeometryHelper::inverse(_itemRotationState));
                updateGL();
                modifiedTheSelection();
                break;
            case SceneTranslation:
            case SceneRotation:
                _additionalSceneTransformation.loadIdentityMatrix();
                updateGL();
                break;
            case None:
                break;
        }

        _isLeftClick = _isRightClick = false;
        _currentTransform = None;
    }

    void GLWidget::translatePoint(int posW, int posH)
    {
        auto point = _scene->selection.selectedPoint;

        HCoordinate3 pointInView = GeometryHelper::transformPoint(*point, _sceneTransformation);
        double planeZ = pointInView.z() / pointInView.w();

        HCoordinate3 transVector =
            modelCoordFromScreen(posW, posH, planeZ) -
            modelCoordFromScreen(_startPosW, _startPosH, planeZ);

        *point += transVector - _prevTransAdded;
        _prevTransAdded = transVector;

        _scene->selection.selectedItem->update();
        updateGL();
        modifiedTheSelection();
    }

    void GLWidget::translateItem(int posW, int posH)
    {
        auto item = _scene->selection.selectedItem;

        HCoordinate3 transVector =
            modelCoordFromScreen(posW, posH, 0) -
            modelCoordFromScreen(_startPosW, _startPosH, 0);

        item->transform(
            GeometryHelper::generateTranslationMatrix(transVector - _prevTransAdded));
        _prevTransAdded = transVector;

        updateGL();
        modifiedTheSelection();
    }

    void GLWidget::rotateItem(int posW, int posH)
    {
        if (!_isInverseActualized)
            updateTransformInverse();

        auto item = _scene->selection.selectedItem;

        DCoordinate3 startCoord = viewCoordFromScreen(_startPosW, _startPosH, 0.0);
        DCoordinate3 current = viewCoordFromScreen(posW, posH, 0.0);

        double radx = (startCoord.y() - current.y()) * PI/4;;
        double rady = (current.x() - startCoord.x()) * PI/4;;

        RealMatrix transform = _sceneTransformation;
        transform =
            GeometryHelper::generateTranslationMatrix(_itemRotationCenterOfGravity) *
            GeometryHelper::rotationAroundOx(radx) *
            GeometryHelper::rotationAroundOy(rady) *
            GeometryHelper::generateTranslationMatrix(-_itemRotationCenterOfGravity) *
            transform;
        transform = _sceneTransformationInverse * transform;

        item->transform(transform * GeometryHelper::inverse(_itemRotationState));
        _itemRotationState = transform;
        modifiedTheSelection();
        updateGL();
    }

    void GLWidget::translateScene(int posW, int posH)
    {
        DCoordinate3 transVector =
            viewCoordFromScreen(posW, posH, 0) -
            viewCoordFromScreen(_startPosW, _startPosH, 0);

        _additionalSceneTransformation = GeometryHelper::generateTranslationMatrix(transVector);
        updateGL();
    }

    void GLWidget::rotateScene(int posW, int posH)
    {
        DCoordinate3 startCoord = viewCoordFromScreen(_startPosW, _startPosH, 0.0);
        DCoordinate3 current = viewCoordFromScreen(posW, posH, 0.0);

        double radx = (startCoord.y() - current.y()) * PI/4;
        double rady = (current.x() - startCoord.x()) * PI/4;

        // around Oy then around Ox
        _additionalSceneTransformation =
            GeometryHelper::rotationAroundOx(radx) *
            GeometryHelper::rotationAroundOy(rady);

        updateGL();
    }

    HCoordinate3 GLWidget::viewCoordFromScreen(int posW, int posH, double planeZ)
    {
        double maxVisibleY = (_eye[2] - planeZ) * std::tan((_fovy / 2.0) * PI / 180);

        return DCoordinate3(
            (1.0*posW/_widgetWidth - 0.5) * maxVisibleY * _aspect / 0.5,
            (-1.0*posH/_widgetHeight + 0.5) * maxVisibleY / 0.5,
            planeZ
        );
    }

    HCoordinate3 GLWidget::modelCoordFromScreen(int posW, int posH, double planeZ)
    {
        if (_isInverseActualized)
            updateTransformInverse();

        return GeometryHelper::transformPoint(
            viewCoordFromScreen(posW, posH, planeZ),
            _sceneTransformationInverse);
    }

    void GLWidget::updateTransformInverse()
    {
        _sceneTransformationInverse = GeometryHelper::inverse(_sceneTransformation);
        _isInverseActualized = true;
    }

    void GLWidget::tryToPickElement(int posW, int posH)
    {
        if (!_isInverseActualized)
            updateTransformInverse();

        HCoordinate3 eyeInScene = GeometryHelper::transformPoint(
            HCoordinate3(_eye[0], _eye[1], _eye[2]),
            _sceneTransformationInverse);

        HCoordinate3 currentPoint = modelCoordFromScreen(posW, posH, 0);

        double zoomFactor = GeometryHelper::transformPoint(
            HCoordinate3(1, 0, 0), _sceneTransformation
        ).length();

        highlight.setNull();

        if (_scene->selection.selectedItem) {
            auto result = _scene->selection.selectedItem->pickControlPointClosestToRay(currentPoint, eyeInScene);
            if (result.distance < 0.08 / zoomFactor) {
                highlight.selectedItem = _scene->selection.selectedItem;
                highlight.selectedPoint = result.point;
            }
        }

        if (!highlight.isPoint()) {
            double closestDist;
            SceneItem *closest = nullptr;

            for (auto item : _scene->items) {
                double currentDist = item->itemDistanceFromPointIfCloseToRay(eyeInScene, currentPoint, 0.2 / zoomFactor);
                if (currentDist == std::numeric_limits<double>::infinity())
                    continue;

                if (!closest || currentDist < closestDist) {
                    closest = item;
                    closestDist = currentDist;
                }
            }

            highlight.selectedItem = closest;
        }
    }

    void GLWidget::updateCursor()
    {
        if (_currentTransform == PointTranslation || _currentTransform == ItemTranslation)
            return setCursor(Qt::ClosedHandCursor);

        if (highlight == _scene->selection && highlight.selectedItem)
            return setCursor(Qt::OpenHandCursor);

        if (_currentTransform == None) {
            if (highlight.isPoint())
                return setCursor(Qt::CrossCursor);
            if (highlight.isWholeItem())
                return setCursor(Qt::PointingHandCursor);
        }

        return setCursor(Qt::ArrowCursor);
    }

    void GLWidget::clearViewTransformations()
    {
        _sceneTransformation.loadIdentityMatrix();
        _sceneTransformationInverse.loadIdentityMatrix();
        _isInverseActualized = true;
        updateGL();
    }

    void GLWidget::moveSelectionToCenter()
    {
        if (!_scene->selection.selectedItem)
            return;

        DCoordinate3 point = _scene->selection.selectedItem->centerOfGravity();
        DCoordinate3 pointInView = GeometryHelper::transformPoint(point, _sceneTransformation);

        _sceneTransformation =
            GeometryHelper::generateTranslationMatrix(-pointInView) *
            _sceneTransformation;
        _isInverseActualized = false;
        updateGL();
    }
}
