#include "CurveSetupTab.h"
#include "ui_CurveSetupTab.h"
#include "CommonUITools.h"
#include "SetupECSpaces/SetupECSpacesDialog.h"
#include "SetupExactDescription/SetupExactDescriptionDialog.h"
#include "Model/CurveItem.h"

#include <QFileDialog>
#include <QDoubleSpinBox>
#include <QPushButton>
#include <QToolButton>
#include <QLabel>
#include <iomanip>
#include <QCheckBox>
#include <fstream>

namespace cagd
{
    CurveSetupTab::CurveSetupTab(QWidget *parent, Scene *scene) :
        QWidget(parent),
        ui(new Ui::CurveSetupTab),
        _scene(scene)
    {
        ui->setupUi(this);

        connect(ui->setupECSpaceButton, SIGNAL(clicked()), this, SLOT(setupECSpaceClicked()));
        connect(ui->exactDescriptionButton, SIGNAL(clicked()), this, SLOT(exactDescriptionClicked()));
        connect(ui->alwaysShowControlPolyCheckBox, SIGNAL(toggled(bool)), this, SLOT(alwaysShowPolygonChanged(bool)));
        connect(ui->showFirstOrderDerivCheckBox, SIGNAL(toggled(bool)), this, SLOT(showFirstDerivChanged(bool)));
        connect(ui->showSecondOrderDerivCheckBox, SIGNAL(toggled(bool)), this, SLOT(showSecondDerivChanged(bool)));
        connect(ui->zerothDerivColorButton, SIGNAL(clicked()), this, SLOT(setZerothDerColor()));
        connect(ui->firstDerivColorButton, SIGNAL(clicked()), this, SLOT(setFirstDerColor()));
        connect(ui->secondDerivColorButton, SIGNAL(clicked()), this, SLOT(setSecondDerColor()));

        connect(ui->oderElevationGoButton, SIGNAL(clicked()), this, SLOT(performOrderElevation()));
        connect(ui->subdivisionGoButton, SIGNAL(clicked()), this, SLOT(performSubdivision()));

        connect(ui->deleteCurveButton, SIGNAL(clicked()), this, SLOT(deleteCurve()));
        connect(ui->saveCurveButton, SIGNAL(clicked()), this, SLOT(saveCurveToFile()));
    }

    CurveSetupTab::~CurveSetupTab()
    {
        delete ui;
    }

    void CurveSetupTab::updateFromScene()
    {
        CurveItem *item = dynamic_cast<CurveItem *>(_scene->selection.selectedItem);

        ui->alwaysShowControlPolyCheckBox->setChecked(item->alwaysShowControlPolygon);
        ui->showFirstOrderDerivCheckBox->setChecked(item->showFirstDerivatives);
        ui->showSecondOrderDerivCheckBox->setChecked(item->showSecondDerivatives);

        double leftEndpoint, rightEndpoint;
        item->bcurve->GetDefinitionDomain(leftEndpoint, rightEndpoint);
        ui->intervalLeftLabel->setText(formatDouble(leftEndpoint).c_str());
        ui->intervalRightLabel->setText(formatDouble(rightEndpoint).c_str());

        ui->subdivisionPointSpinBox->setMinimum(leftEndpoint + 0.0001);
        ui->subdivisionPointSpinBox->setMaximum(rightEndpoint - 0.0001);
        ui->subdivisionPointSpinBox->setValue((leftEndpoint + rightEndpoint) / 2);
    }

    void CurveSetupTab::setupECSpaceClicked()
    {
        CurveItem *item = dynamic_cast<CurveItem *>(_scene->selection.selectedItem);
        if (!item) return;

        auto *dialog = new SetupECSpacesDialog(
            nullptr,
            _scene->ecSpaceDatabase,
            item->ecSpace);

        bool result = dialog->exec();
        delete dialog;
        activateMainGLWidget();

        if (result)
        {
            item->regenerate();
            configChanged();
        }
    }

    void CurveSetupTab::exactDescriptionClicked()
    {
        CurveItem *item = dynamic_cast<CurveItem *>(_scene->selection.selectedItem);
        if (!item) return;

        std::vector<DCoordinate3> result;
        SetupExactDescriptionDialog dialog(nullptr, &result, item->ecSpace);

        if (dialog.exec() == QDialog::Accepted) {
            item->updateControlPointsForExactDescription(result);
            configChanged();
        }
    }

    void CurveSetupTab::alwaysShowPolygonChanged(bool value)
    {
        CurveItem *item = dynamic_cast<CurveItem *>(_scene->selection.selectedItem);

        if (item->alwaysShowControlPolygon != value) {
            item->alwaysShowControlPolygon = value;
            configChanged();
        }
    }

    void CurveSetupTab::showFirstDerivChanged(bool value)
    {
        CurveItem *item = dynamic_cast<CurveItem *>(_scene->selection.selectedItem);

        if (item->showFirstDerivatives != value) {
            item->showFirstDerivatives = value;
            configChanged();
        }
    }

    void CurveSetupTab::showSecondDerivChanged(bool value)
    {
        CurveItem *item = dynamic_cast<CurveItem *>(_scene->selection.selectedItem);

        if (item->showSecondDerivatives != value) {
            item->showSecondDerivatives = value;
            configChanged();
        }
    }

    void CurveSetupTab::setZerothDerColor()
    {
        CurveItem *item = dynamic_cast<CurveItem *>(_scene->selection.selectedItem);

        if (CommonUITools::changeColorViaPopup(item->color))
            configChanged();
    }

    void CurveSetupTab::setFirstDerColor()
    {
        CurveItem *item = dynamic_cast<CurveItem *>(_scene->selection.selectedItem);

        if (CommonUITools::changeColorViaPopup(item->derivColor))
            configChanged();
    }

    void CurveSetupTab::setSecondDerColor()
    {
        CurveItem *item = dynamic_cast<CurveItem *>(_scene->selection.selectedItem);

        if (CommonUITools::changeColorViaPopup(item->secondDerivColor))
            configChanged();
    }

    void CurveSetupTab::performOrderElevation()
    {
        CurveItem *item = dynamic_cast<CurveItem *>(_scene->selection.selectedItem);

        item->performOrderElevation(
            ui->zeroRealSpinBox->value(),
            ui->zeroImagSpinBox->value(),
            ui->multiplicitySpinBox->value());
        _scene->selection.selectedPoint = nullptr;
        configChanged();
    }

    void CurveSetupTab::performSubdivision()
    {
        CurveItem *item = dynamic_cast<CurveItem *>(_scene->selection.selectedItem);

        item->performSubdivision(ui->subdivisionPointSpinBox->value());
        _scene->selection.selectedPoint = nullptr;
        configChanged();
    }

    std::string CurveSetupTab::formatDouble(double value) const
    {
        std::stringstream ss;
        ss << std::fixed << std::setprecision(4) << value;
        return ss.str();
    }

    void CurveSetupTab::deleteCurve()
    {
        _scene->deleteItem(_scene->selection.selectedItem);
        configChanged();
    }

    void CurveSetupTab::saveCurveToFile()
    {
        CurveItem *item = dynamic_cast<CurveItem *>(_scene->selection.selectedItem);

        auto fileName = QFileDialog::getSaveFileName(
            this,
            "Select File to Save",
            "");

        if (fileName.length() > 0) {
            std::ofstream fout(fileName.toUtf8().constData());
            fout << *item;
        }
    }
}
