#include "ECSpace.h"
#include <cmath>
#include "../Utils/DoubleEquality.h"
#include "../Core/RealMatrices.h"
#include "../Core/RealMatrixDecompositions.h"

namespace cagd
{
    ECSpace::ECSpace():
        definitionDomain(0, 1),
        characteristicPolynomial(),
        _rho(0,0),
        _mu(0,0)
    {}

    ECSpace::ECSpace(const ECSpace &ecSpace):
        definitionDomain(ecSpace.definitionDomain),
        characteristicPolynomial(ecSpace.characteristicPolynomial),
        _rho(1,1),
        _mu(1,1)
    {}

    const ECSpace &ECSpace::operator =(const ECSpace &rightSide)
    {
        if(this != &rightSide) {
           definitionDomain = rightSide.definitionDomain;
           characteristicPolynomial = rightSide.characteristicPolynomial;
        }
        return *this;
    }

    void ECSpace::preprocessing()
    {
        _dimension = characteristicPolynomial.getGrade();
        _isReflectionInvariant = characteristicPolynomial.isOddOrEvenFuncion();

        setupOrdBases();
        calculateRho();
        calculateMuAndLambda();
    }

    void ECSpace::setupOrdBases()
    {
        ordBases.resize(_dimension);

        unsigned ordBaseIndex = 0;
        for (auto zero : characteristicPolynomial.zeros) {
            for (int i=0; i<zero.multiplicity; ++i) {
                if (doubleEquals(zero.absImaginary, 0)) {
                    ordBases[ordBaseIndex++] = OrdinaryBasisFunction(i, zero.real);
                }
                else {
                    ordBases[ordBaseIndex++] = OrdinaryBasisFunction(i, zero.real, zero.absImaginary, false);
                    ordBases[ordBaseIndex++] = OrdinaryBasisFunction(i, zero.real, zero.absImaginary, true);
                }
            }
        }
    }

    void ECSpace::calculateRho()
    {
        _rho.resizeColumns(_dimension);
        _rho.resizeRows(_dimension);
        for (unsigned vInd = 0; vInd < _dimension; ++vInd)
        {
            unsigned currLine = 0;
            RealMatrix equation(_dimension, _dimension);

            for (unsigned order = 0; order < vInd; ++order) {
                for (unsigned k = 0; k < _dimension; ++k)
                    equation(currLine, k) = ordBases[k].getDerivative(order, definitionDomain.first);
                ++currLine;
            }

            for (unsigned k = 0; k < _dimension; ++k)
                equation(currLine, k) = ordBases[k].getDerivative(vInd, definitionDomain.first);
            ++currLine;

            for (unsigned order = 0; order < _dimension - 1 - vInd; ++order) {
                for (unsigned k = 0; k < _dimension; ++k)
                    equation(currLine, k) = ordBases[k].getDerivative(order, definitionDomain.second);
                ++currLine;
            }

            ColumnMatrix<double> unknowns(_dimension);
            ColumnMatrix<double> rhs(_dimension);

            for (unsigned i = 0; i < _dimension; ++i)
                rhs[i] = 0;
            rhs[vInd] = 1;

            PLUDecomposition decomp(equation);
            decomp.solveLinearSystem(rhs, unknowns);

            for (unsigned i = 0; i < _dimension; ++i)
                _rho(vInd,i) = unknowns[i];
        }
    }

    void ECSpace::calculateMuAndLambda()
    {
        RealMatrix revWronskianBeta(_dimension, _dimension);

        for (unsigned order = 0; order < _dimension; ++order) {
            for (int vInd = _dimension - 1; vInd >= 0; --vInd) {
                double currV = 0;

                for (unsigned k = 0; k < _dimension; ++k)
                    currV += _rho(vInd, k) * ordBases[k].getDerivative(order, definitionDomain.second);

                revWronskianBeta(order, _dimension-1-vInd) = currV;
            }
        }

        auto decomp = FactorizedUnpivotedLUDecomposition(revWronskianBeta);

        RealMatrix eye(_dimension, _dimension);
        eye.loadIdentityMatrix();

        RealMatrix Linv(_dimension, _dimension);
        PLUDecomposition ldecomp(decomp.L());
        ldecomp.solveLinearSystem(eye, Linv);

        _lambda.resize(_dimension);
        for (unsigned i = 0; i < _dimension; ++i)
            _lambda[i] = Linv(i, 0);

        _mu.resizeColumns(_dimension);
        _mu.resizeRows(_dimension);

        PLUDecomposition udecomp(decomp.U());
        udecomp.solveLinearSystem(eye, _mu);
    }


    unsigned ECSpace::getDimension() const
    {
        return _dimension;
    }

    double ECSpace::NNBBaseDerivative(unsigned index, unsigned deriv_order, double input) const
    {
        double sgn = 1;
        double sum = 0;
        unsigned i = _dimension - 1 - index;

        if (_isReflectionInvariant && index < i) {
            i = index;
            sgn = (deriv_order % 2) ? -1 : 1;
            input = definitionDomain.first + definitionDomain.second - input;
        }

        for (unsigned r = 0; r <= i; ++r) {
            double innerSum = 0;

            for (unsigned k = 0; k < _dimension; ++k)
                innerSum += _rho(_dimension-1-r, k) * ordBases[k].getDerivative(deriv_order, input);

            sum += _mu(r,i) * innerSum;
        }

        return sgn * _lambda[i] * sum;
    }

    ECSpace *ECSpace::getAugmentedECSpace(CharacteristicPolynomial::Zero zeroToAdd) const
    {
        ECSpace *newECSpace = new ECSpace(*this);

        bool found = false;
        for (auto &zero : newECSpace->characteristicPolynomial.zeros)
            if (doubleEquals(zero.real, zeroToAdd.real) && doubleEquals(zero.absImaginary, zeroToAdd.absImaginary)) {
                zero.multiplicity += zeroToAdd.multiplicity;
                found = true;
            }

        if (!found) {
            CharacteristicPolynomial::Zero z;
            z.real = zeroToAdd.real;
            z.absImaginary = zeroToAdd.absImaginary;
            z.multiplicity = zeroToAdd.multiplicity;
            newECSpace->characteristicPolynomial.zeros.push_back(z);
        }

        return newECSpace;
    }

    RealMatrix ECSpace::getBasisTransformationMatrix() const
    {
        unsigned n = _dimension - 1;

        RealMatrix ordBasisDerivativesAlpha(_dimension, n/2 + 1);
        RealMatrix ordBasisDerivativesBeta(_dimension, n/2 + 1);
        for (unsigned i = 0; i <= n; ++i)
            for (unsigned order = 0; order <= n/2; ++order) {
                ordBasisDerivativesAlpha(i, order) = ordBases[i].getDerivative(order, definitionDomain.first);
                ordBasisDerivativesBeta(i, order) = ordBases[i].getDerivative(order, definitionDomain.second);
            }

        RealMatrix nnbBasisDerivativesAlpha(_dimension, n/2 + 1);
        RealMatrix nnbBasisDerivativesBeta(_dimension, n/2 + 1);
        for (unsigned i = 0; i <= n; ++i)
            for (unsigned order = 0; order <= n/2; ++order) {
                nnbBasisDerivativesAlpha(i, order) = NNBBaseDerivative(i, order, definitionDomain.first);
                nnbBasisDerivativesBeta(i, order) =  NNBBaseDerivative(i, order, definitionDomain.second);
            }

        RealMatrix t(_dimension, _dimension);

        for (unsigned i = 0; i <= n; ++i) {
            t(i, 0) = ordBasisDerivativesAlpha(i, 0);
            for (unsigned j = 1; j <= n/2; ++j) {
                double sum = 0;
                for (unsigned k = 0; k < j; ++k)
                    sum += t(i, k) * nnbBasisDerivativesAlpha(k, j);

                t(i, j) = (ordBasisDerivativesAlpha(i, j) - sum) / nnbBasisDerivativesAlpha(j, j);
            }

            t(i, n) = ordBasisDerivativesBeta(i, 0);
            for (unsigned j = 1; j <= (n-1)/2; ++j) {
                double sum = 0;
                for (unsigned k = 0; k < j; ++k)
                    sum += t(i, n-k) * nnbBasisDerivativesBeta(n-k, j);

                t(i, n-j) = (ordBasisDerivativesBeta(i, j) - sum) / nnbBasisDerivativesBeta(n-j, j);
            }
        }

        return t;
    }

    std::ostream &operator<<(std::ostream &stream, const ECSpace &space)
    {
        return stream
            << space.definitionDomain.first << ' ' << space.definitionDomain.second << std::endl
            << space.characteristicPolynomial;
    }

    std::istream &operator>>(std::istream &stream, ECSpace &space)
    {
        return stream
            >> space.definitionDomain.first >> space.definitionDomain.second
            >> space.characteristicPolynomial;
    }
}

