#include "CharacteristicPolynomial.h"

#include "../Utils/DoubleEquality.h"

namespace cagd
{

    CharacteristicPolynomial::CharacteristicPolynomial():
        zeros {{0, 0, 1}}
    {}

    CharacteristicPolynomial::CharacteristicPolynomial(const CharacteristicPolynomial &polynomial):
        zeros(polynomial.zeros)
    {}

    const CharacteristicPolynomial &CharacteristicPolynomial::operator= (const CharacteristicPolynomial &rightSide)
    {
        if (this != &rightSide)
            zeros = rightSide.zeros;
        return *this;
    }

    int CharacteristicPolynomial:: getGrade()
    {
        int total = 0;

        for (auto z : zeros)
            total += (doubleEquals(z.absImaginary, 0) ? 1 : 2) * z.multiplicity;

        return total;
    }

    std::vector<double> CharacteristicPolynomial::getCoefficients()
    {
        int grade = getGrade();
        std::vector<double> coefficients(grade + 1, 0);
        coefficients[0] = 1;

        for (auto zero : zeros) {
            if (doubleEquals(zero.absImaginary, 0))
                for (int times = 0; times < zero.multiplicity; ++times)
                    multiply(coefficients, { -zero.real, 1.0 });
            else
                for (int times = 0; times < zero.multiplicity; ++times)
                    multiply(coefficients, {
                        std::pow(zero.real, 2) + std::pow(zero.absImaginary, 2),
                        - 2 * zero.real,
                        1.0 });
        }

        return coefficients;
    }

    bool CharacteristicPolynomial::isOddOrEvenFuncion()
    {
        for (unsigned i = 0; i < zeros.size(); ++i) {
            if (doubleEquals(zeros[i].real, 0)) continue;

            bool found = false;
            for (unsigned j = 0; j < zeros.size(); ++j) {
                if (doubleEquals(-zeros[i].real, zeros[j].real) &&
                    doubleEquals(zeros[i].absImaginary, zeros[j].absImaginary) &&
                    zeros[i].multiplicity == zeros[j].multiplicity)
                {
                    found = true;
                }
            }

            if (!found)
                return false;
        }

        return true;
    }

    void CharacteristicPolynomial::multiply(std::vector<double> &destination, std::initializer_list<double> multiplier)
    {
        std::vector<double> old(destination.size(), 0);
        old.swap(destination);

        unsigned offset = 0;
        for (double coefficient : multiplier) {
            for (unsigned i = offset; i < destination.size(); ++i)
                destination[i] += old[i-offset] * coefficient;
            offset++;
        }
    }

    bool CharacteristicPolynomial::Zero::operator ==(const CharacteristicPolynomial::Zero &rightSide) const
    {
        return doubleEquals(real, rightSide.real) &&
            doubleEquals(absImaginary, rightSide.absImaginary) &&
            multiplicity == rightSide.multiplicity;
    }

    bool CharacteristicPolynomial::Zero::operator !=(const CharacteristicPolynomial::Zero &rightSide) const
    {
        return !((*this) == rightSide);
    }

    std::ostream &operator<<(std::ostream &stream, const CharacteristicPolynomial &polynomial)
    {
        stream << polynomial.zeros.size() << std::endl;
        for (auto zero : polynomial.zeros)
            stream << zero.real << ' ' << zero.absImaginary << ' ' << zero.multiplicity << std::endl;
        return stream;
    }

    std::istream &operator>>(std::istream &stream, CharacteristicPolynomial &polynomial)
    {
        int size;
        stream >> size;
        polynomial.zeros.resize(size);
        for (auto &zero : polynomial.zeros)
            stream >> zero.real >> zero.absImaginary >> zero.multiplicity;
        return stream;
    }
}
