#include "BSurface3.h"
#include "BCurve3.h"
#include <cmath>

namespace cagd
{

    BSurface3::BSurface3(const ECSpace *ecSpaceU, const ECSpace *ecSpaceV):
        TensorProductSurface3(
            ecSpaceU->definitionDomain.first,
            ecSpaceU->definitionDomain.second,
            ecSpaceV->definitionDomain.first,
            ecSpaceV->definitionDomain.second,
            ecSpaceU->getDimension(),
            ecSpaceV->getDimension()
        ),
        _ecSpaceU(ecSpaceU),
        _ecSpaceV(ecSpaceV),
        _cache_u_count(0),
        _cache_v_count(0)
    {}

    GLboolean BSurface3::UBlendingFunctionValues(GLdouble u_knot, RowMatrix<GLdouble> &blending_values)
    {
        unsigned dim = _ecSpaceU->getDimension();
        blending_values.resizeColumns(dim);

        for (unsigned k = 0; k < dim; ++k)
            blending_values[k] = getNNBDerivativeU(k, 0, u_knot);

        return true;
    }

    GLboolean BSurface3::VBlendingFunctionValues(GLdouble v_knot, RowMatrix<GLdouble> &blending_values)
    {
        unsigned dim = _ecSpaceV->getDimension();
        blending_values.resizeColumns(dim);

        for (unsigned k = 0; k < dim; ++k)
            blending_values[k] = getNNBDerivativeV(k, 0, v_knot);

        return true;
    }

    GLboolean BSurface3::CalculatePartialDerivatives(GLuint maximum_order_of_partial_derivatives,
                                                     GLdouble u, GLdouble v,
                                                     TensorProductSurface3::PartialDerivatives &pd)
    {
        pd.resizeRows(maximum_order_of_partial_derivatives + 1);

        for (unsigned i = 0; i <= maximum_order_of_partial_derivatives; ++i)
            for (unsigned j = 0; j <= i; ++j) {
                pd(i,j) = DCoordinate3(0, 0, 0);
                for (unsigned uPoint = 0; uPoint < _ecSpaceU->getDimension(); ++uPoint)
                    for (unsigned vPoint = 0; vPoint < _ecSpaceV->getDimension(); ++vPoint)
                        pd(i,j) += (*this)(uPoint, vPoint)
                            * getNNBDerivativeU(uPoint, i-j, u)
                            * getNNBDerivativeV(vPoint, j, v);
            }

        return true;
    }

    std::pair<ECSpace *, BSurface3 *> BSurface3::performUOrderElevation(
        double reZero, double imZero, int multiplicity) const
    {
        CharacteristicPolynomial::Zero z;
        z.real = reZero;
        z.absImaginary = imZero;
        z.multiplicity = multiplicity;

        ECSpace *newECSpaceU = _ecSpaceU->getAugmentedECSpace(z);
        newECSpaceU->preprocessing();
        BSurface3 *newSurface = new BSurface3(newECSpaceU, _ecSpaceV);

        BCurve3 *oldCurve = new BCurve3(_ecSpaceU);
        BCurve3 *newCurve = new BCurve3(newECSpaceU);

        for (unsigned colInd = 0; colInd < _ecSpaceV->getDimension(); ++colInd) {
            for (unsigned rowInd = 0; rowInd < _ecSpaceU->getDimension(); ++rowInd)
                (*oldCurve)[rowInd] = (*this)(rowInd, colInd);
            oldCurve->calculateDataForOrderElevatedCurve(newECSpaceU, newCurve);
            for (unsigned rowInd = 0; rowInd < newECSpaceU->getDimension(); ++rowInd)
                (*newSurface)(rowInd, colInd) = (*newCurve)[rowInd];
        }

        delete newCurve;
        delete oldCurve;

        return std::make_pair(newECSpaceU, newSurface);
    }

    std::pair<ECSpace *, BSurface3 *> BSurface3::performVOrderElevation(
        double reZero, double imZero, int multiplicity) const
    {
        CharacteristicPolynomial::Zero z;
        z.real = reZero;
        z.absImaginary = imZero;
        z.multiplicity = multiplicity;

        ECSpace *newECSpaceV = _ecSpaceV->getAugmentedECSpace(z);
        newECSpaceV->preprocessing();
        BSurface3 *newSurface = new BSurface3(_ecSpaceU, newECSpaceV);

        BCurve3 *oldCurve = new BCurve3(_ecSpaceV);
        BCurve3 *newCurve = new BCurve3(newECSpaceV);

        for (unsigned rowInd = 0; rowInd < _ecSpaceU->getDimension(); ++rowInd) {
            for (unsigned colInd = 0; colInd < _ecSpaceV->getDimension(); ++colInd)
                (*oldCurve)[colInd] = (*this)(rowInd, colInd);
            oldCurve->calculateDataForOrderElevatedCurve(newECSpaceV, newCurve);
            for (unsigned colInd = 0; colInd < newECSpaceV->getDimension(); ++colInd)
                (*newSurface)(rowInd, colInd) = (*newCurve)[colInd];
        }

        delete newCurve;
        delete oldCurve;

        return std::make_pair(newECSpaceV, newSurface);
    }

    BSurface3::SubdivisionResult BSurface3::performUSubdivision(double gamma) const
    {
        SubdivisionResult result;

        double alpha = _ecSpaceU->definitionDomain.first;
        double beta = _ecSpaceU->definitionDomain.second;

        if (gamma <= alpha || gamma >= beta)
        {
            result.ecSpaceLeft = result.ecSpaceRight = nullptr;
            result.bSurfaceLeft = result.bSurfaceRight = nullptr;
            return result;
        }

        result.ecSpaceLeft = new ECSpace(*_ecSpaceU);
        result.ecSpaceLeft->definitionDomain = std::make_pair(alpha, gamma);
        result.ecSpaceLeft->preprocessing();
        result.bSurfaceLeft = new BSurface3(result.ecSpaceLeft, _ecSpaceV);

        result.ecSpaceRight = new ECSpace(*_ecSpaceU);
        result.ecSpaceRight->definitionDomain = std::make_pair(gamma, beta);
        result.ecSpaceRight->preprocessing();
        auto spaceCopy = new ECSpace(*_ecSpaceV);
        spaceCopy->preprocessing();
        result.bSurfaceRight = new BSurface3(result.ecSpaceRight, spaceCopy);

        BCurve3 *old = new BCurve3(_ecSpaceU);
        BCurve3::SubdivisionResult curvesResult;
        curvesResult.ecSpaceLeft = result.ecSpaceLeft;
        curvesResult.ecSpaceRight = result.ecSpaceRight;
        curvesResult.bCurveLeft = new BCurve3(result.ecSpaceLeft);
        curvesResult.bCurveRight = new BCurve3(result.ecSpaceRight);

        for (unsigned col = 0; col < _ecSpaceV->getDimension(); ++col) {
            for (unsigned row = 0; row < _ecSpaceU->getDimension(); ++row)
                (*old)[row] = (*this)(row, col);
            old->fillSubdivisionValues(curvesResult, alpha, gamma, beta);
            for (unsigned row = 0; row < _ecSpaceU->getDimension(); ++row) {
                (*result.bSurfaceLeft)(row, col) = (*curvesResult.bCurveLeft)[row];
                (*result.bSurfaceRight)(row, col) = (*curvesResult.bCurveRight)[row];
            }
        }

        delete curvesResult.bCurveLeft;
        delete curvesResult.bCurveRight;
        return result;
    }

    BSurface3::SubdivisionResult BSurface3::performVSubdivision(double gamma) const
    {
        SubdivisionResult result;

        double alpha = _ecSpaceV->definitionDomain.first;
        double beta = _ecSpaceV->definitionDomain.second;

        if (gamma <= alpha || gamma >= beta)
        {
            result.ecSpaceLeft = result.ecSpaceRight = nullptr;
            result.bSurfaceLeft = result.bSurfaceRight = nullptr;
            return result;
        }

        result.ecSpaceLeft = new ECSpace(*_ecSpaceV);
        result.ecSpaceLeft->definitionDomain = std::make_pair(alpha, gamma);
        result.ecSpaceLeft->preprocessing();
        result.bSurfaceLeft = new BSurface3(_ecSpaceU, result.ecSpaceLeft);

        result.ecSpaceRight = new ECSpace(*_ecSpaceV);
        result.ecSpaceRight->definitionDomain = std::make_pair(gamma, beta);
        result.ecSpaceRight->preprocessing();
        auto spaceCopy = new ECSpace(*_ecSpaceU);
        spaceCopy->preprocessing();
        result.bSurfaceRight = new BSurface3(spaceCopy, result.ecSpaceRight);

        BCurve3 *old = new BCurve3(_ecSpaceV);
        BCurve3::SubdivisionResult curvesResult;
        curvesResult.ecSpaceLeft = result.ecSpaceLeft;
        curvesResult.ecSpaceRight = result.ecSpaceRight;
        curvesResult.bCurveLeft = new BCurve3(result.ecSpaceLeft);
        curvesResult.bCurveRight = new BCurve3(result.ecSpaceRight);

        for (unsigned row = 0; row < _ecSpaceU->getDimension(); ++row) {
            for (unsigned col = 0; col < _ecSpaceV->getDimension(); ++col)
                (*old)[col] = (*this)(row, col);
            old->fillSubdivisionValues(curvesResult, alpha, gamma, beta);
            for (unsigned col = 0; col < _ecSpaceV->getDimension(); ++col) {
                (*result.bSurfaceLeft)(row, col) = (*curvesResult.bCurveLeft)[col];
                (*result.bSurfaceRight)(row, col) = (*curvesResult.bCurveRight)[col];
            }
        }

        delete curvesResult.bCurveLeft;
        delete curvesResult.bCurveRight;
        return result;
    }

    void BSurface3::updateControlPointsForExactDescription(const std::vector<DCoordinate3> &coefficients)
    {
        unsigned dimensionU = _ecSpaceU->getDimension();
        RealMatrix transformU = _ecSpaceU->getBasisTransformationMatrix();

        unsigned dimensionV = _ecSpaceV->getDimension();
        RealMatrix transformV = _ecSpaceV->getBasisTransformationMatrix();

        for (unsigned i = 0; i < dimensionU; ++i)
            for (unsigned j = 0; j < dimensionV; ++j)
                _data(i, j) = DCoordinate3(0, 0, 0);

        for (unsigned i = 0; i < dimensionU; ++i)
            for (unsigned j = 0; j < dimensionV; ++j) {
                const DCoordinate3 &coef = coefficients[i * dimensionV + j];

                for (unsigned k = 0; k < dimensionU; ++k)
                    for (unsigned l = 0; l < dimensionV; ++l)
                        _data(k, l) += coef * transformU(i, k) * transformV(j, l);
            }

    }

    TriangulatedMesh3* BSurface3::GenerateImage(
        GLuint u_div_point_count, GLuint v_div_point_count,
        GLenum usage_flag)
    {
        if (u_div_point_count != _cache_u_count) {
            _u_derivative_cache.clear();
            _cache_u_count = u_div_point_count;

            double alpha = _ecSpaceU->definitionDomain.first;
            double beta = _ecSpaceU->definitionDomain.second;

            for (unsigned index = 0; index < _ecSpaceU->getDimension(); ++index) {
                for (unsigned order = 0; order <= _ecSpaceU->getDimension(); ++order) {
                    for (unsigned ui = 0; ui < _cache_u_count; ++ui) {
                        _u_derivative_cache[getKey(index, order, ui)] =
                           _ecSpaceU->NNBBaseDerivative(index, order, alpha + ui * (beta - alpha) / _cache_u_count);
                    }
                }
            }

        }

        if (v_div_point_count != _cache_v_count) {
            _v_derivative_cache.clear();
            _cache_v_count = v_div_point_count;

            double alpha = _ecSpaceV->definitionDomain.first;
            double beta = _ecSpaceV->definitionDomain.second;

            for (unsigned index = 0; index < _ecSpaceV->getDimension(); ++index) {
                for (unsigned order = 0; order <= _ecSpaceV->getDimension(); ++order) {
                    for (unsigned vi = 0; vi < _cache_v_count; ++vi) {
                        _v_derivative_cache[getKey(index, order, vi)] =
                           _ecSpaceV->NNBBaseDerivative(index, order, alpha + vi * (beta - alpha) / _cache_v_count);
                    }
                }
            }
        }

        return TensorProductSurface3::GenerateImage(u_div_point_count, v_div_point_count, usage_flag);
    }

    double BSurface3::getNNBDerivativeU(unsigned index, unsigned order, double u)
    {
        double alpha = _ecSpaceU->definitionDomain.first;
        double beta = _ecSpaceU->definitionDomain.second;

        unsigned uind = (unsigned) std::lrint((u - alpha) / (beta-alpha) * _cache_u_count);

        auto it = _u_derivative_cache.find(getKey(index, order, uind));
        if (it != _u_derivative_cache.end())
            return it->second;
        else return _ecSpaceU->NNBBaseDerivative(index, order, u);
    }

    double BSurface3::getNNBDerivativeV(unsigned index, unsigned order, double v)
    {
        double alpha = _ecSpaceV->definitionDomain.first;
        double beta = _ecSpaceV->definitionDomain.second;

        unsigned vind = (unsigned) std::lrint((v - alpha) / (beta-alpha) * _cache_v_count);

        auto it = _v_derivative_cache.find(getKey(index, order, vind));
        if (it != _v_derivative_cache.end())
            return it->second;
        else return _ecSpaceV->NNBBaseDerivative(index, order, v);
    }

    unsigned BSurface3::getKey(unsigned index, unsigned order, unsigned knot)
    {
        return (index << 20) + (order << 12) + knot;
    }
}
