#include "BCurve3.h"
#include "../Utils/DoubleEquality.h"

#include <vector>
#include <iostream>

namespace cagd
{
    BCurve3::BCurve3(const ECSpace* ecSpace):
        LinearCombination3(
            ecSpace->definitionDomain.first,
            ecSpace->definitionDomain.second,
            ecSpace->getDimension()
        ),
        _ecSpace(ecSpace)
    {}

    GLboolean BCurve3::BlendingFunctionValues(GLdouble u, RowMatrix<GLdouble>& values) const
    {
        unsigned dim = _ecSpace->getDimension();
        values.resizeColumns(dim);

        for (unsigned k = 0; k < dim; ++k)
            values[k] = _ecSpace->NNBBaseDerivative(k, 0, u);

        return true;
    }

    GLboolean BCurve3::CalculateDerivatives(GLuint max_order_of_derivatives, GLdouble u, Derivatives& d) const
    {
        d.resizeRows(max_order_of_derivatives + 1);

        for (unsigned ord = 0; ord <= max_order_of_derivatives; ++ord ) {
            DCoordinate3 sum(0,0,0);

            for(unsigned i = 0; i < _data.rowCount(); ++i)
                sum += _data[i] * _ecSpace->NNBBaseDerivative(i, ord, u);

            d[ord] = sum;
        }

        return true;
    }

    std::pair<ECSpace*, BCurve3*> BCurve3::performOrderElevation(double reZero, double imZero, int multiplicity) const
    {
        CharacteristicPolynomial::Zero z;
        z.real = reZero;
        z.absImaginary = imZero;
        z.multiplicity = multiplicity;

        ECSpace *newECSpace = _ecSpace->getAugmentedECSpace(z);
        newECSpace->preprocessing();

        BCurve3 * newCurve = new BCurve3(newECSpace);

        if (!calculateDataForOrderElevatedCurve(newECSpace, newCurve)) {
            delete newCurve, newCurve = nullptr;
            delete newECSpace, newECSpace = nullptr;
        }

        return std::make_pair(newECSpace, newCurve);
    }

    bool BCurve3::calculateDataForOrderElevatedCurve(ECSpace *newECSpace, BCurve3 *newCurve) const
    {
        int oldDim = _ecSpace->getDimension();
        int newDim = newECSpace->getDimension();

        if (newDim - oldDim == 1) {
            unsigned n = oldDim - 1;
            double alpha = _ecSpace->definitionDomain.first;
            double beta = _ecSpace->definitionDomain.second;

            (*newCurve)[0] = (*this)[0];
            (*newCurve)[n+1] = (*this)[n];

            for (unsigned i=1; i<=n/2; ++i) {
                double xi =
                    _ecSpace->NNBBaseDerivative(i, i, alpha) /
                    newECSpace->NNBBaseDerivative(i, i, alpha);

                (*newCurve)[i] = (1-xi) * (*this)[i-1] + xi * (*this)[i];
            }

            for (unsigned i=1; i<=(n+1)/2; ++i) {
                double zeta =
                    _ecSpace->NNBBaseDerivative(n-i, i, beta) /
                    newECSpace->NNBBaseDerivative(n+1-i, i, beta);

                (*newCurve)[n+1-i] = zeta * (*this)[n-i] + (1-zeta) * (*this)[n+1-i];
            }
        }
        else { // difference in dimension >= 2

            ColumnMatrix<GLdouble>    knot_vector(newDim);
            ColumnMatrix<DCoordinate3>  sample(newDim);

            GLboolean sampling_aborted = GL_FALSE;
            GLdouble  step = (_ecSpace->definitionDomain.second - _ecSpace->definitionDomain.first) / (newDim - 1);

            #pragma omp parallel for
            for (int i = 0; i < newDim; i++) {
                #pragma omp flush (sampling_aborted)
                if (!sampling_aborted)
                {
                    // uniform subdivision points of the interval $\left[\alpha,\beta\right]$
                    knot_vector[i] = std::min(
                        _ecSpace->definitionDomain.first + i * step,
                        _ecSpace->definitionDomain.second);

                    Derivatives d;
                    if (!CalculateDerivatives(0, knot_vector[i], d))
                    {
                        sampling_aborted = GL_TRUE;
                        #pragma omp flush (sampling_aborted)
                    }
                    else
                    {
                        sample[i] = d[0]; // sample point of the original curve
                    }
                }
            }

            if (sampling_aborted || !newCurve->UpdateDataForInterpolation(knot_vector, sample))
            {
                return false;
            }
        }

        return true;
    }

    BCurve3::SubdivisionResult BCurve3::performSubdivision(double gamma) const
    {
        SubdivisionResult result;

        double alpha = _ecSpace->definitionDomain.first;
        double beta = _ecSpace->definitionDomain.second;

        if (gamma <= alpha || gamma >= beta)
        {
            result.ecSpaceLeft = result.ecSpaceRight = nullptr;
            result.bCurveLeft = result.bCurveRight = nullptr;
            return result;
        }

        result.ecSpaceLeft = new ECSpace(*_ecSpace);
        result.ecSpaceLeft->definitionDomain = std::make_pair(alpha, gamma);
        result.ecSpaceLeft->preprocessing();
        result.bCurveLeft = new BCurve3(result.ecSpaceLeft);

        result.ecSpaceRight = new ECSpace(*_ecSpace);
        result.ecSpaceRight->definitionDomain = std::make_pair(gamma, beta);
        result.ecSpaceRight->preprocessing();
        result.bCurveRight = new BCurve3(result.ecSpaceRight);

        fillSubdivisionValues(result, alpha, gamma, beta);

        return result;
    }

    void BCurve3::fillSubdivisionValues(
        BCurve3::SubdivisionResult &result,
        double alpha, double gamma, double beta) const
    {
        unsigned n = _ecSpace->getDimension() - 1;

        Derivatives der_alpha, der_gamma, der_beta;
        this->CalculateDerivatives(n/2, alpha, der_alpha);
        this->CalculateDerivatives(n/2, gamma, der_gamma);
        this->CalculateDerivatives(n/2, beta, der_beta);

        std::vector<DCoordinate3> lambda(n+1), rho(n+1);
        lambda[0] = (*this)[0];
        lambda[n] = rho[0] = der_gamma[0];
        rho[n] = (*this)[n];

        for (unsigned i = 1; i <= (n-1)/2; ++i)
        {
            DCoordinate3 sum_prev;
            for (unsigned j = 0; j < i; ++j)
                sum_prev += lambda[j] * result.ecSpaceLeft->NNBBaseDerivative(j, i, alpha);

            lambda[i] =
                1.0 / result.ecSpaceLeft->NNBBaseDerivative(i, i, alpha) *
                (der_alpha[i] - sum_prev);
        }

        for (unsigned i = 1; i <= n/2; ++i)
        {
            DCoordinate3 sum_prev;
            for (unsigned j = 0; j < i; ++j)
                sum_prev += lambda[n-j] * result.ecSpaceLeft->NNBBaseDerivative(n-j, i, gamma);

            lambda[n-i] =
                1.0 / result.ecSpaceLeft->NNBBaseDerivative(n-i, i, gamma) *
                (der_gamma[i] - sum_prev);
        }

        for (unsigned i = 1; i <= n/2; ++i)
        {
            DCoordinate3 sum_prev;
            for (unsigned j = 0; j < i; ++j)
                sum_prev += rho[j] * result.ecSpaceRight->NNBBaseDerivative(j, i, gamma);

            rho[i] =
                1.0 / result.ecSpaceRight->NNBBaseDerivative(i, i, gamma) *
                (der_gamma[i] - sum_prev);
        }

        for (unsigned i = 1; i <= (n-1)/2; ++i)
        {
            DCoordinate3 sum_prev;
            for (unsigned j = 0; j < i; ++j)
                sum_prev += rho[n-j] * result.ecSpaceRight->NNBBaseDerivative(n-j, i, beta);

            rho[n-i] =
                1.0 / result.ecSpaceRight->NNBBaseDerivative(n-i, i, beta) *
                (der_beta[i] - sum_prev);
        }

        for (unsigned i = 0; i <= n; ++i)
            (*result.bCurveLeft)[i] = lambda[i];

        for (unsigned i = 0; i <= n; ++i)
            (*result.bCurveRight)[i] = rho[i];
    }

    void BCurve3::updateControlPointsForExactDescription(const std::vector<DCoordinate3> &coefficients)
    {
        RealMatrix transform = _ecSpace->getBasisTransformationMatrix();
        unsigned dimension = _ecSpace->getDimension();

        for (unsigned j = 0; j < dimension; ++j) {
            _data[j] = DCoordinate3(0, 0, 0);
            for (unsigned i = 0; i < dimension; ++i)
                _data[j] += coefficients[i] * transform(i, j);
        }
    }
}
