#include <algorithm>

#include "TensorProductSurfaces3.h"
#include "RealMatrices.h"
#include "RealMatrixDecompositions.h"

using namespace cagd;
using namespace std;

// special constructor
TensorProductSurface3::PartialDerivatives::PartialDerivatives(GLuint maximum_order_of_partial_derivatives):
        TriangularMatrix<DCoordinate3>(maximum_order_of_partial_derivatives + 1)
{
}

// generates the image (i.e., the approximating triangulated mesh) of the tensor product surface
TriangulatedMesh3* TensorProductSurface3::GenerateImage(GLuint u_div_point_count, GLuint v_div_point_count, GLenum usage_flag)
{
    if (u_div_point_count <= 1 || v_div_point_count <= 1)
        return GL_FALSE;

    // calculating number of vertices, unit normal vectors and texture coordinates
    GLuint vertex_count = u_div_point_count * v_div_point_count;

    // calculating number of triangular faces
    GLuint face_count = 2 * (u_div_point_count - 1) * (v_div_point_count - 1);

    TriangulatedMesh3 *result = nullptr;
    result = new TriangulatedMesh3(vertex_count, face_count, usage_flag);

    if (!result)
        return nullptr;

    // uniform subdivision grid in the definition domain
    GLdouble du = (_u_max - _u_min) / (u_div_point_count - 1);
    GLdouble dv = (_v_max - _v_min) / (v_div_point_count - 1);

    // uniform subdivision grid in the unit square
    GLfloat sdu = 1.0f / ((float) u_div_point_count - 1.0f);
    GLfloat tdv = 1.0f / ((float) v_div_point_count - 1.0f);

    // for face indexing
    GLuint current_face = 0;

    Matrix<PartialDerivatives> pdMatrix(u_div_point_count, v_div_point_count);

    // calculating all needed surface data
    #pragma omp parallel for
    for (int i = 0; i < (int) u_div_point_count; ++i)
    {
        GLdouble u = _u_min + i * du;

        #pragma omp parallel for
        for (int j = 0; j < (int) v_div_point_count; ++j)
        {
             GLdouble v = _v_min + j * dv;
             CalculatePartialDerivatives(1, u, v, pdMatrix(i,j));
        }
    }

    for (GLuint i = 0; i < u_div_point_count; ++i)
    {
        GLfloat  s = (GLfloat) i * sdu;
        for (GLuint j = 0; j < v_div_point_count; ++j)
        {
            GLfloat  t = (GLfloat) j * tdv;

            /*
                3-2
                |/|
                0-1
            */
            GLuint index[4];

            index[0] = i * v_div_point_count + j;
            index[1] = index[0] + 1;
            index[2] = index[1] + v_div_point_count;
            index[3] = index[2] - 1;

            auto &pd = pdMatrix(i, j);

            // surface point
            (*result)._vertex[index[0]] = pd(0, 0);

            // unit surface normal
            (*result)._normal[index[0]] = pd(1, 0);
            (*result)._normal[index[0]] ^= pd(1, 1);
            (*result)._normal[index[0]].normalize();

            // texture coordinates
            (*result)._tex[index[0]].s() = s;
            (*result)._tex[index[0]].t() = t;

            // faces
            if (i < u_div_point_count - 1 && j < v_div_point_count - 1)
            {
                (*result)._face[current_face][0] = index[0];
                (*result)._face[current_face][1] = index[1];
                (*result)._face[current_face][2] = index[2];
                ++current_face;

                (*result)._face[current_face][0] = index[0];
                (*result)._face[current_face][1] = index[2];
                (*result)._face[current_face][2] = index[3];
                ++current_face;
            }
        }
    }

    return result;
}

// ensures interpolation, i.e. s(u_i, v_j) = d_{i,j}
GLboolean TensorProductSurface3::UpdateDataForInterpolation(const RowMatrix<GLdouble>& u_knot_vector, const ColumnMatrix<GLdouble>& v_knot_vector, Matrix<DCoordinate3>& data_points_to_interpolate)
{
    GLuint row_count = _data.rowCount();
    if (!row_count)
        return GL_FALSE;

    GLuint column_count = _data.columnCount();
    if (!column_count)
        return GL_FALSE;

    if (u_knot_vector.columnCount() != row_count || v_knot_vector.rowCount() != column_count || data_points_to_interpolate.rowCount() != row_count || data_points_to_interpolate.columnCount() != column_count)
        return GL_FALSE;

    // 1: calculate the u-collocation matrix and perfom LU-decomposition on it
    RowMatrix<GLdouble> u_blending_values;

    RealMatrix u_collocation_matrix(row_count, row_count);

    for (GLuint i = 0; i < row_count; ++i)
    {
        if (!UBlendingFunctionValues(u_knot_vector(i), u_blending_values))
            return GL_FALSE;
        u_collocation_matrix.setRow(i, u_blending_values);
    }

    PLUDecomposition udecomp(u_collocation_matrix);

    // 2: calculate the v-collocation matrix and perform LU-decomposition on it
    RowMatrix<GLdouble> v_blending_values;

    RealMatrix v_collocation_matrix(column_count, column_count);

    for (GLuint j = 0; j < column_count; ++j)
    {
        if (!VBlendingFunctionValues(v_knot_vector(j), v_blending_values))
            return GL_FALSE;
        v_collocation_matrix.setRow(j, v_blending_values);
    }

    PLUDecomposition vdecomp(v_collocation_matrix);

    // 3:   for all fixed j in {0, 1,..., column_count} determine control points
    //
    //      a_k(v_j) = sum_{l=0}^{column_count} _data(l, j) G_l(v_j), k = 0, 1,..., row_count
    //
    //      such that
    //
    //      sum_{k=0}^{row_count} a_k(v_j) F_k(u_i) = data_points_to_interpolate(i, j),
    //
    //      for all i = 0, 1,..., row_count.
    Matrix<DCoordinate3> a(row_count, column_count);
    if (!udecomp.solveLinearSystem(data_points_to_interpolate, a))
        return GL_FALSE;

    // 4:   for all fixed i in {0, 1,..., row_count} determine control point
    //
    //      _data[i][j], j = 0, 1,..., column_count
    //
    //      such that
    //
    //      sum_{l=0}^{column_count} _data(i, l) G_l(v_j) = a_i(v_j)
    //
    //      for all j = 0, 1,..., column_count.
    if (!vdecomp.solveLinearSystem(a, _data, GL_FALSE))
        return GL_FALSE;

    return GL_TRUE;
}


// initializes all partial derivatives to the origin
GLvoid TensorProductSurface3::PartialDerivatives::LoadNullVectors()
{
    for(auto &line : _data)
        for(auto &ref : line)
            ref = DCoordinate3(0,0,0);
}


// special constructor
TensorProductSurface3::TensorProductSurface3(
        GLdouble u_min, GLdouble u_max,
        GLdouble v_min, GLdouble v_max,
        GLuint row_count, GLuint column_count,
        GLboolean u_closed, GLboolean v_closed):
    _u_min(u_min),
    _u_max(u_max),
    _v_min(v_min),
    _v_max(v_max),
    _data(row_count,column_count),
    _u_closed(u_closed),
    _v_closed(v_closed)
{}


// copy constructor
TensorProductSurface3::TensorProductSurface3(const TensorProductSurface3& surface):
    _u_min(surface._u_min),
    _u_max(surface._u_max),
    _v_min(surface._v_min),
    _v_max(surface._v_max),
    _data(surface._data),
    _u_closed(surface._u_closed),
    _v_closed(surface._v_closed)
{}


// assignment operator
TensorProductSurface3& TensorProductSurface3::operator =(const TensorProductSurface3& surface)
{
    if(this != &surface)
    {
        _u_min = surface._u_min;
        _u_max = surface._u_max;
        _v_min = surface._v_min;
        _v_max = surface._v_max;
        _data = surface._data;
        _u_closed = surface._u_closed;
        _v_closed = surface._v_closed;
    }
    return *this;
}

// destructor
TensorProductSurface3::~TensorProductSurface3()
{
    DeleteVertexBufferObjectsOfData();
}

// set/get the definition domain of the surface
GLvoid TensorProductSurface3::SetUInterval(GLdouble u_min, GLdouble u_max)
{
    _u_min = u_min;
    _u_max = u_max;
}
GLvoid TensorProductSurface3::SetVInterval(GLdouble v_min, GLdouble v_max)
{
    _v_min = v_min;
    _v_max = v_max;
}

GLvoid TensorProductSurface3::GetUInterval(GLdouble& u_min, GLdouble& u_max) const
{
    u_min = _u_min;
    u_max = _u_max;
}
GLvoid TensorProductSurface3::GetVInterval(GLdouble& v_min, GLdouble& v_max) const
{
    v_min = _v_min;
    v_max = _v_max;
}

GLuint TensorProductSurface3::GetUControlPointCount() const
{
    return _data.rowCount();
}

GLuint TensorProductSurface3::GetVControlPointCount() const
{
    return _data.columnCount();
}

// set coordinates of a selected data point
GLboolean TensorProductSurface3::SetData(GLuint row, GLuint column, GLdouble x, GLdouble y, GLdouble z)
{
    if(row < _data.rowCount() && column < _data.columnCount()){
        _data(row,column) = DCoordinate3(x,y,z);
        return GL_TRUE;
    }
    else return GL_FALSE;
}
GLboolean TensorProductSurface3::SetData(GLuint row, GLuint column, const DCoordinate3& point)
{
    if(row < _data.rowCount() && column < _data.columnCount()){
        _data(row,column) = point;
        return GL_TRUE;
    }
    else return GL_FALSE;
}

// get coordinates of a selected data point
GLboolean TensorProductSurface3::GetData(GLuint row, GLuint column, GLdouble& x, GLdouble& y, GLdouble& z) const
{
    if(row < _data.rowCount() && column < _data.columnCount()){
        x = _data(row,column).x();
        y = _data(row,column).y();
        z = _data(row,column).z();
        return GL_TRUE;
    }
    else return GL_FALSE;
}
GLboolean TensorProductSurface3::GetData(GLuint row, GLuint column, DCoordinate3& point) const
{
    if(row < _data.rowCount() && column < _data.columnCount()){
        point = _data(row,column);
        return GL_TRUE;
    }
    else return GL_FALSE;
}


// get data by value
DCoordinate3 TensorProductSurface3::operator ()(GLuint row, GLuint column) const
{
    return _data(row,column);
}

// get data by reference
DCoordinate3& TensorProductSurface3::operator ()(GLuint row, GLuint column)
{
    return _data(row,column);
}


// VBO handling methods
GLvoid TensorProductSurface3::DeleteVertexBufferObjectsOfData()
{
    if (_vbo_data)
    {
        glDeleteBuffers(1, &_vbo_data);
        _vbo_data = 0;
    }
}


GLboolean TensorProductSurface3::UpdateVertexBufferObjectsOfData(GLenum usage_flag)
{
    GLuint data_count = _data.rowCount() * _data.columnCount();
    if (!data_count)
        return GL_FALSE;

    if (usage_flag != GL_STREAM_DRAW  && usage_flag != GL_STREAM_READ  && usage_flag != GL_STREAM_COPY
     && usage_flag != GL_DYNAMIC_DRAW && usage_flag != GL_DYNAMIC_READ && usage_flag != GL_DYNAMIC_COPY
     && usage_flag != GL_STATIC_DRAW  && usage_flag != GL_STATIC_READ  && usage_flag != GL_STATIC_COPY)
        return GL_FALSE;

    DeleteVertexBufferObjectsOfData();

    glGenBuffers(1, &_vbo_data);
    if (!_vbo_data)
        return GL_FALSE;

    glBindBuffer(GL_ARRAY_BUFFER, _vbo_data);
    glBufferData(GL_ARRAY_BUFFER, 2 * data_count * 3 * sizeof(GLfloat), 0, usage_flag);

    GLfloat *coordinate = (GLfloat*)glMapBuffer(GL_ARRAY_BUFFER, GL_WRITE_ONLY);
    if (!coordinate)
    {
        glBindBuffer(GL_ARRAY_BUFFER, 0);
        DeleteVertexBufferObjectsOfData();
        return GL_FALSE;
    }

    for (GLuint i = 0; i < _data.rowCount(); ++i)
    {
        for (GLuint j = 0; j < _data.columnCount(); ++j)
        {
            DCoordinate3 &cp = _data(i,j);
            for (GLuint c = 0; c < 3; ++c)
            {
                *coordinate = (GLfloat)cp[c];
                ++coordinate;
            }
        }
    }

    for (GLuint j = 0; j < _data.columnCount(); ++j)
    {
        for (GLuint i = 0; i < _data.rowCount(); ++i)
        {
            DCoordinate3 &cp = _data(i,j);
            for (GLuint c = 0; c < 3; ++c)
            {
                *coordinate = (GLfloat)cp[c];
                ++coordinate;
            }
        }
    }

    if (!glUnmapBuffer(GL_ARRAY_BUFFER))
    {
        glBindBuffer(GL_ARRAY_BUFFER, 0);
        DeleteVertexBufferObjectsOfData();
        return GL_FALSE;
    }

    glBindBuffer(GL_ARRAY_BUFFER, 0);

    return GL_TRUE;
}


GLboolean TensorProductSurface3::RenderData(GLenum render_mode) const
{
    if (!_vbo_data)
        return GL_FALSE;

    if (render_mode != GL_LINE_STRIP && render_mode != GL_LINE_LOOP && render_mode != GL_POINTS)
        return GL_FALSE;

    glEnableClientState(GL_VERTEX_ARRAY);
        glBindBuffer(GL_ARRAY_BUFFER, _vbo_data);
            glVertexPointer(3, GL_FLOAT, 0, (const GLvoid*)0);

            if(render_mode == GL_POINTS)
            {
                glDrawArrays(render_mode, 0, _data.rowCount() * _data.columnCount());
            }
            else
            {
                GLuint offset = 0;
                for(GLuint i=0; i<_data.rowCount(); ++i, offset+=_data.columnCount()){
                    glDrawArrays(render_mode, offset, _data.columnCount());
                }
                for(GLuint j=0; j<_data.columnCount(); ++j, offset+=_data.rowCount()){
                    glDrawArrays(render_mode, offset, _data.rowCount());
                }
            }
        glBindBuffer(GL_ARRAY_BUFFER, 0);
    glDisableClientState(GL_VERTEX_ARRAY);

    return GL_TRUE;
}


// generate u-directional isoparametric lines
RowMatrix<GenericCurve3*>* TensorProductSurface3::GenerateUIsoparametricLines(GLuint iso_line_count,
                                                      GLuint maximum_order_of_derivatives,
                                                      GLuint div_point_count,
                                                      GLenum usage_flag)
{
    if(div_point_count < 2 || iso_line_count < 2){
        return nullptr;
    }

    RowMatrix<GenericCurve3*> *result = new RowMatrix<GenericCurve3*>(iso_line_count);
    if(!result) return nullptr;

    GLdouble v_step = (_v_max - _v_min) / (iso_line_count - 1);
    GLdouble u_step = (_u_max - _u_min) / (div_point_count - 1);

    for(GLuint i=0; i<iso_line_count; ++i){
        (*result)[i] = new GenericCurve3(maximum_order_of_derivatives,div_point_count,usage_flag);

        if(!(*result)[i]){
            for(GLuint j=0; j<i; ++j) delete (*result)[j];
            delete result;
            return result = nullptr;
        }

        GLdouble v = std::min(_v_min + i*v_step, _v_max);

        for(GLuint j=0; j<div_point_count; ++j){
            GLdouble u = min(_u_min + j*u_step, _u_max);

            PartialDerivatives pd;

            if(!CalculatePartialDerivatives(maximum_order_of_derivatives,u,v,pd)){
                //clean-up
            }

            for(GLuint r = 0; r<=maximum_order_of_derivatives; ++r){
                (*(*result)[i])(r,j) = pd(r,0);
            }
        }
    }

    return result;
}

// generate v-directional isoparametric lines
RowMatrix<GenericCurve3*>* TensorProductSurface3::GenerateVIsoparametricLines(GLuint iso_line_count,
                                                      GLuint maximum_order_of_derivatives,
                                                      GLuint div_point_count,
                                                      GLenum usage_flag)
{
    if(div_point_count < 2 || iso_line_count < 2){
        return nullptr;
    }

    RowMatrix<GenericCurve3*> *result = new RowMatrix<GenericCurve3*>(iso_line_count);
    if(!result) return nullptr;

    GLdouble u_step = (_u_max - _u_min) / (iso_line_count - 1);
    GLdouble v_step = (_v_max - _v_min) / (div_point_count - 1);

    for(GLuint i=0; i<iso_line_count; ++i){
        (*result)[i] = new GenericCurve3(maximum_order_of_derivatives,div_point_count,usage_flag);

        if(!(*result)[i]){
            for(GLuint j=0; j<i; ++j) delete (*result)[j];
            delete result;
            return result = nullptr;
        }

        GLdouble u = std::min(_u_min + i*u_step, _u_max);

        for(GLuint j=0; j<div_point_count; ++j){
            GLdouble v = min(_v_min + j*v_step, _v_max);

            PartialDerivatives pd;

            if(!CalculatePartialDerivatives(maximum_order_of_derivatives,u,v,pd)){
                //clean-up
            }

            for(GLuint r = 0; r<=maximum_order_of_derivatives; ++r){
                (*(*result)[i])(r,j) = pd(r,0);
            }
        }
    }

    return result;
}
