#pragma once

#include <algorithm>
#include <cassert>
#include <iostream>
#include <vector>

namespace cagd
{
    // forward declaration of the template class Matrix
    template <typename T>
    class Matrix;

    // forward declaration of the template class RowMatrix
    template <typename T>
    class RowMatrix;

    // forward declaration of the template class ColumnMatrix
    template <typename T>
    class ColumnMatrix;

    // forward declaration of the template class TriangularMatrix
    template <typename T>
    class TriangularMatrix;

    // forward declarations of overloaded and templated input/output from/to stream operators
    template <typename T>
    std::ostream& operator << (std::ostream& lhs, const Matrix<T>& rhs);

    template <typename T>
    std::istream& operator >>(std::istream& lhs, Matrix<T>& rhs);

    template <typename T>
    std::ostream& operator << (std::ostream& lhs, const TriangularMatrix<T>& rhs);

    template <typename T>
    std::istream& operator >>(std::istream& lhs, TriangularMatrix<T>& rhs);

    // Interface of the template class Matrix
    template <typename T>
    class Matrix
    {
        friend std::ostream& cagd::operator << <T>(std::ostream&, const Matrix<T>& rhs);
        friend std::istream& cagd::operator >> <T>(std::istream&, Matrix<T>& rhs);

    protected:
        unsigned            _row_count;
        unsigned            _column_count;
        std::vector<T> _data;

    public:
        // default/special constructor
        explicit Matrix(int row_count = 1, int column_count = 1);

        // get element by non-constant reference
        T& operator ()(unsigned row, unsigned column);

        // get element by constant reference
        const T& operator ()(unsigned row, unsigned column) const;

        // get dimensions
        unsigned rowCount() const;
        unsigned columnCount() const;

        // set dimensions
        virtual bool resizeRows(unsigned row_count);
        virtual bool resizeColumns(unsigned column_count);

        // update
        bool setRow(unsigned index, const RowMatrix<T>& row);
        bool setColumn(unsigned index, const ColumnMatrix<T>& column);

        // clone function required by smart pointers based on the deep copy ownership policy
        virtual Matrix* clone() const;

        // destructor
        virtual ~Matrix();
    };

    // Interface of the template class RowMatrix
    template <typename T>
    class RowMatrix: public Matrix<T>
    {
    public:
        // default/special constructor
        explicit RowMatrix(int column_count = 1);

        // get element by non-constant reference
        T& operator ()(unsigned column);
        T& operator [](unsigned column);

        // get element by constant reference
        const T& operator ()(unsigned column) const;
        const T& operator [](unsigned column) const;

        // a row matrix consists of a single row
        bool resizeRows(int row_count);

        // redeclared clone function required by smart pointers based on the deep copy ownership policy
        RowMatrix* clone() const;
    };

    // Interface of the template class ColumnMatrix:
    template <typename T>
    class ColumnMatrix: public Matrix<T>
    {
    public:
        // default/special constructor
        explicit ColumnMatrix(int row_count = 1);

        // get element by non-constant reference
        T& operator ()(unsigned row);
        T& operator [](unsigned row);

        // get element by constant reference
        const T& operator ()(unsigned row) const;
        const T& operator [](unsigned row) const;

        // a column matrix consists of a single column
        bool resizeColumns(unsigned column_count);

        // redeclared clone function required by smart pointers based on the deep copy ownership policy
        ColumnMatrix* clone() const;
    };

    // Interface of the template class TriangularMatrix
    template <typename T>
    class TriangularMatrix
    {
        friend std::ostream& cagd::operator << <T>(std::ostream&,
                                                   const TriangularMatrix<T>& rhs);
        friend std::istream& cagd::operator >> <T>(std::istream&,
                                                   TriangularMatrix<T>& rhs);

    protected:
        int                           _row_count;
        std::vector< std::vector<T> > _data;

    public:
        // default/special constructor
        explicit TriangularMatrix(int row_count = 1);

        // get element by non-constant reference
        T& operator ()(int row, int column);

        // get element by constant reference
        const T& operator ()(int row, int column) const;

        // get dimension
        int rowCount() const;

        // set dimension
        bool resizeRows(int row_count);

        // clone function required by smart pointers based on the deep copy ownership policy
        virtual TriangularMatrix* clone() const;

        // destructor
        virtual ~TriangularMatrix();
    };

    // Implementation of the template class Matrix:

    // default/special constructor
    template <typename T>
    Matrix<T>::Matrix(int row_count, int column_count):
        _row_count(row_count < 0 || column_count <= 0 ? 0 : row_count),
        _column_count(column_count < 0 || !_row_count ? 0 : column_count),
        _data(_row_count * _column_count)
    {
        assert("The row count of a matrix should be non-negative!" && row_count >= 0);
        assert("The column count of a matrix should be non-negative!" &&
               column_count >= 0);
    }

    // get element by non-constant reference
    template <typename T>
    inline T& Matrix<T>::operator ()(unsigned row, unsigned column)
    {
        assert("The given row index is out of bounds!" && (row >= 0 && row < _row_count));
        assert("The given column index is out of bounds!" &&
               (column >= 0 && column < _column_count));
        return _data[row * _column_count + column];
    }

    // get element by constant reference
    template <typename T>
    inline const T& Matrix<T>::operator ()(unsigned row, unsigned column) const
    {
        assert("The given row index is out of bounds!" && (row >= 0 && row < _row_count));
        assert("The given column index is out of bounds!" &&
               (column >= 0 && column < _column_count));
        return _data[row * _column_count + column];
    }

    // get dimensions
    template <typename T>
    inline unsigned Matrix<T>::rowCount() const
    {
        return _row_count;
    }

    template <typename T>
    inline unsigned Matrix<T>::columnCount() const
    {
        return _column_count;
    }

    // set dimensions
    template <typename T>
    bool Matrix<T>::resizeRows(unsigned row_count)
    {
        assert("The row count of a matrix should be non-negative!" && row_count >= 0);

        if (row_count < 0)
        {
            return false;
        }

        if (_row_count != row_count)
        {
            _data.resize(row_count * _column_count);
            _row_count = row_count;

            if (_row_count == 0)
            {
                _column_count = 0;
            }
        }

        return false;
    }

    template <typename T>
    bool Matrix<T>::resizeColumns(unsigned column_count)
    {
        if (column_count < 0)
        {
            return false;
        }

        if (_column_count != column_count)
        {
            std::vector<T> new_data(_row_count * column_count);

            int preserved_column_count = std::min(column_count, _column_count);

            #pragma omp parallel for
            for (int r = 0; r < (int) _row_count; r++)
            {
                for (int c = 0,
                     old_offset = r * _column_count,
                     new_offset = r * column_count;
                     c < preserved_column_count;
                     c++)
                {
                    int old_index       = old_offset + c;
                    int new_index       = new_offset + c;
                    new_data[new_index] = _data[old_index];
                }
            }

            _data.swap(new_data);

            _column_count = column_count;

            if (_column_count == 0)
            {
                _row_count = 0;
            }
        }

        return true;
    }

    // update
    template <class T>
    bool Matrix<T>::setRow(unsigned row, const RowMatrix<T>& entire_row)
    {
        assert("The given column index is out of bounds!" &&
               (row >= 0 && row < _row_count));
        assert("The column counts of the underlying row matrices should be equal!" &&
                _column_count == entire_row.columnCount());

        if (row < 0 || row >= _row_count || _column_count != entire_row.columnCount())
        {
            return false;
        }

        int offset = row * _column_count;

        #pragma omp parallel for
        for (int c = 0; c < (int) _column_count; c++)
        {
            _data[offset + c] = entire_row._data[c];
        }

        return true;
    }

    template <class T>
    bool Matrix<T>::setColumn(unsigned column, const ColumnMatrix<T>& entire_column)
    {
        assert("The given column index is out of bounds!" &&
               (column >= 0 && column < _column_count));
        assert("The row counts of the underlying column matrices should be equal!" &&
               _row_count == entire_column.rowCount());

        if (column < 0 || column >= _column_count ||
            _row_count != entire_column.rowCount())
        {
            return false;
        }

        #pragma omp parallel for
        for (int r = 0; r < (int) _row_count; r++)
        {
            _data[r * _column_count + column] = entire_column._data[r];
        }

        return true;
    }

    // clone function required by smart pointers based on the deep copy ownership policy
    template <typename T>
    Matrix<T>* Matrix<T>::clone() const
    {
        return new (std::nothrow) Matrix<T>(*this);
    }

    // destructor
    template <typename T>
    Matrix<T>::~Matrix()
    {
        _row_count = _column_count = 0;
        //_data.clear();
    }

    // Implementation of the template class RowMatrix:

    // default/special constructor
    template <typename T>
    RowMatrix<T>::RowMatrix(int column_count): Matrix<T>(1, column_count)
    {
    }

    // get element by non-constant reference
    template <typename T>
    inline T& RowMatrix<T>::operator ()(unsigned column)
    {
        assert("The given column index is out of bounds!" &&
               (column >= 0 && column < this->_column_count));
        return this->_data[column];
    }

    template <typename T>
    inline T& RowMatrix<T>::operator [](unsigned column)
    {
        assert("The given column index is out of bounds!" &&
               (column >= 0 && column < this->_column_count));
        return this->_data[column];
    }

    // get element by constant reference
    template <typename T>
    inline const T& RowMatrix<T>::operator ()(unsigned column) const
    {
        assert("The given column index is out of bounds!" &&
               (column >= 0 && column < this->_column_count));
        return this->_data[column];
    }

    template <typename T>
    inline const T& RowMatrix<T>::operator [](unsigned column) const
    {
        assert("The given column index is out of bounds!" &&
               (column >= 0 && column < this->_column_count));
        return this->_data[column];
    }

    // a row matrix consists of a single row
    template <typename T>
    bool RowMatrix<T>::resizeRows(int row_count)
    {
        return (row_count == 1);
    }

    // redefined clone function required by smart pointers based on the deep copy ownership policy
    template <typename T>
    RowMatrix<T>* RowMatrix<T>::clone() const
    {
        return new (std::nothrow) RowMatrix<T>(*this);
    }

    // Implementation of the template class ColumnMatrix:

    // default/special constructor
    template <typename T>
    ColumnMatrix<T>::ColumnMatrix(int row_count): Matrix<T>(row_count, 1)
    {
    }

    // get element by non-constant reference
    template <typename T>
    inline T& ColumnMatrix<T>::operator ()(unsigned row)
    {
        assert("The given row index is out of bounds!" &&
               (row >= 0 && row < this->_row_count));
        return this->_data[row];
    }

    template <typename T>
    inline T& ColumnMatrix<T>::operator [](unsigned row)
    {
        assert("The given row index is out of bounds!" &&
               (row >= 0 && row < this->_row_count));
        return this->_data[row];
    }

    // get element by constant reference
    template <typename T>
    inline const T& ColumnMatrix<T>::operator ()(unsigned row) const
    {
        assert("The given row index is out of bounds!" &&
               (row >= 0 && row < this->_row_count));
        return this->_data[row];
    }

    template <typename T>
    inline const T& ColumnMatrix<T>::operator [](unsigned row) const
    {
        assert("The given row index is out of bounds!" &&
               (row >= 0 && row < this->_row_count));
        return this->_data[row];
    }

    // a column matrix consists of a single column
    template <typename T>
    bool ColumnMatrix<T>::resizeColumns(unsigned column_count)
    {
        return (column_count == 1);
    }

    // redefined clone function required by smart pointers based on the deep copy ownership policy
    template <typename T>
    ColumnMatrix<T>* ColumnMatrix<T>::clone() const
    {
        return new (std::nothrow) ColumnMatrix<T>(*this);
    }

    // Implementation of the template class TriangularMatrix:

    // default/special constructor
    template <typename T>
    TriangularMatrix<T>::TriangularMatrix(int row_count):
        _row_count(row_count < 0 ? 0 : row_count),
        _data(_row_count)
    {
        assert("The row count of a triangular matrix should be a non-negative "
               "integer!" && row_count >= 0);

        for (int r = 0; r < _row_count; r++)
        {
            _data[r].resize(r + 1);
        }
    }

    // get element by non-constant reference
    template <typename T>
    inline T& TriangularMatrix<T>::operator ()(int row, int column)
    {
        assert("The given row index is out of bounds!" && (row >= 0 && row < _row_count));
        assert("The given column index is out of bounds!" &&
               (column >= 0 && column <= row));
        return _data[row][column];
    }

    // get element by constant reference
    template <typename T>
    inline const T& TriangularMatrix<T>::operator ()(int row, int column) const
    {
        assert("The given row index is out of bounds!" && (row >= 0 && row < _row_count));
        assert("The given column index is out of bounds!" &&
               (column >= 0 && column <= row));
        return _data[row][column];
    }

    // get dimension
    template <typename T>
    inline int TriangularMatrix<T>::rowCount() const
    {
        return _row_count;
    }

    // set dimension
    template <typename T>
    bool TriangularMatrix<T>::resizeRows(int row_count)
    {
        assert("The row count of a triangular matrix should be a non-negative "
               "integer!" && row_count >= 0);

        if (row_count < 0)
        {
            return false;
        }

        if (_row_count != row_count)
        {

            _data.resize(row_count);

            for (int r = _row_count; r < row_count; r++)
            {
                _data[r].resize(r + 1);
            }

            _row_count = row_count;
        }

        return true;
    }

    // clone function required by smart pointers based on the deep copy ownership policy
    template <typename T>
    TriangularMatrix<T>* TriangularMatrix<T>::clone() const
    {
        return new (std::nothrow) TriangularMatrix<T>(*this);
    }

    // destructor
    template <typename T>
    TriangularMatrix<T>::~TriangularMatrix()
    {
        _row_count = 0;
        _data.clear();
    }

    // Definitions of overloaded and templated input/output from/to stream operators:

    // output to stream
    template <typename T>
    std::ostream& operator <<(std::ostream& lhs, const Matrix<T>& rhs)
    {
        lhs << rhs._row_count << " " << rhs._column_count << std::endl;

        for (unsigned r = 0; r < rhs._row_count; r++)
        {
            for (unsigned c = 0, offset = r * rhs._column_count; c < rhs._column_count; c++)
            {
                lhs << rhs._data[offset + c] << " ";
            }

            lhs << std::endl;
        }

        return lhs;
    }

    // input from stream
    template <typename T>
    std::istream& operator >>(std::istream& lhs, Matrix<T>& rhs)
    {
        lhs >> rhs._row_count >> rhs._column_count;

        int size = rhs._row_count * rhs._column_count;

        rhs._data.resize(size);

        for (int i = 0; i < size; i++)
        {
            lhs >> rhs._data[i];
        }

        return lhs;
    }

    // output to stream
    template <typename T>
    std::ostream& operator <<(std::ostream& lhs, const TriangularMatrix<T>& rhs)
    {
        lhs << rhs._row_count << std::endl;

        for (int r = 0; r < rhs._row_count; r++)
        {
            for (int c = 0; c <= r; c++)
            {
                lhs << rhs._data[r][c] << " ";
            }
            lhs << std::endl;
        }

        return lhs;
    }

    // input from stream
    template <typename T>
    std::istream& operator >>(std::istream& lhs, TriangularMatrix<T>& rhs)
    {
        lhs >> rhs._row_count;

        rhs._data.resize(rhs._row_count);
        for (int r = 0; r < rhs._row_count; r++)
        {
            rhs._data[r].resize(r + 1);

            for (int c = 0; c <= r; r++)
            {
                lhs >> rhs._data[r][c];
            }
        }

        return lhs;
    }
}
