#pragma once

#include <cmath>
#include <GL/glew.h>
#include <iostream>
#include "DCoordinates3.h"

namespace cagd
{
    //--------------------------------------
    // 3-dimensional homogeneous coordinates
    //--------------------------------------
    class HCoordinate3
    {
    protected:
        GLfloat _data[4]; // x, y, z, w;

    public:
        // default constructor
        HCoordinate3();

        // special constructor
        HCoordinate3(GLfloat x, GLfloat y, GLfloat z = 0.0, GLfloat w = 1.0);
        HCoordinate3(const DCoordinate3 &point);

        GLfloat operator[](GLuint rhs) const;
        GLfloat x() const;
        GLfloat y() const;
        GLfloat z() const;
        GLfloat w() const;

        GLfloat& operator[](GLuint rhs);
        GLfloat& x();
        GLfloat& y();
        GLfloat& z();
        GLfloat& w();

        // add
        const HCoordinate3 operator +(const HCoordinate3& rhs) const;

        HCoordinate3& operator +=(const HCoordinate3& rhs);
        const HCoordinate3 operator -(const HCoordinate3& rhs) const;
        HCoordinate3& operator -=(const HCoordinate3& rhs);

        // dot product
        GLfloat operator *(const HCoordinate3& rhs) const;

        // cross product
        const HCoordinate3 operator ^(const HCoordinate3& rhs) const;
        HCoordinate3& operator ^=(const HCoordinate3& rhs);

        const HCoordinate3 operator *(GLfloat rhs) const;
        HCoordinate3& operator *=(GLfloat rhs);
        const HCoordinate3 operator /(GLfloat rhs) const;
        HCoordinate3& operator /=(GLfloat rhs);

        // length of vector represented by this homogeneous coordinate
        GLfloat length() const;

        // normalize
        HCoordinate3& normalize();

        operator DCoordinate3() const;
    };

    // default constructor
    inline HCoordinate3::HCoordinate3()
    {
        _data[0] = _data[1] = _data[2] = 0.0;
        _data[3] = 1.0;
    }

    // special constructor
    inline HCoordinate3::HCoordinate3(GLfloat x, GLfloat y, GLfloat z, GLfloat w)
    {
        _data[0] = x;
        _data[1] = y;
        _data[2] = z;
        _data[3] = w;
    }

    inline HCoordinate3::HCoordinate3(const DCoordinate3 &point)
    {
        _data[0] = (GLfloat) point[0];
        _data[1] = (GLfloat) point[1];
        _data[2] = (GLfloat) point[2];
        _data[3] = (GLfloat) 1.0;
    }

    // add
    inline const HCoordinate3 HCoordinate3::operator +(const HCoordinate3& rhs) const
    {
        return HCoordinate3(
                rhs.w() * x() + w() * rhs.x(),
                rhs.w() * y() + w() * rhs.y(),
                rhs.w() * z() + w() * rhs.z(),
                w() * rhs.w());
    }

    inline GLfloat HCoordinate3::operator[](GLuint rhs) const
    {
        return _data[rhs];
    }
    inline GLfloat HCoordinate3::x() const
    {
        return _data[0];
    }
    inline GLfloat HCoordinate3::y() const
    {
        return _data[1];
    }
    inline GLfloat HCoordinate3::z() const
    {
        return _data[2];
    }
    inline GLfloat HCoordinate3::w() const
    {
        return _data[3];
    }

    inline GLfloat& HCoordinate3::operator[](GLuint rhs)
    {
        return _data[rhs];
    }
    inline GLfloat& HCoordinate3::x()
    {
        return _data[0];
    }
    inline GLfloat& HCoordinate3::y()
    {
        return _data[1];
    }
    inline GLfloat& HCoordinate3::z()
    {
        return _data[2];
    }
    inline GLfloat& HCoordinate3::w()
    {
        return _data[3];
    }

    inline HCoordinate3& HCoordinate3::operator +=(const HCoordinate3& rhs)
    {
           _data[0] = _data[0] * rhs._data[3] + rhs._data[0] * _data[3];
           _data[1] = _data[1] * rhs._data[3] + rhs._data[1] * _data[3];
           _data[2] = _data[2] * rhs._data[3] + rhs._data[2] * _data[3];
           _data[3] *= rhs._data[3];
           return *this;
    }

    inline const HCoordinate3 HCoordinate3::operator -(const HCoordinate3& rhs) const
    {
        return HCoordinate3(
                rhs.w() * x() - w() * rhs.x(),
                rhs.w() * y() - w() * rhs.y(),
                rhs.w() * z() - w() * rhs.z(),
                w() * rhs.w());
    }

    inline HCoordinate3& HCoordinate3::operator -=(const HCoordinate3& rhs)
    {
        _data[0] = _data[0] * rhs._data[3] - rhs._data[0] * _data[3];
        _data[1] = _data[1] * rhs._data[3] - rhs._data[1] * _data[3];
        _data[2] = _data[2] * rhs._data[3] - rhs._data[2] * _data[3];
        _data[3] *= rhs._data[3];
        return *this;
    }

    inline GLfloat HCoordinate3::operator *(const HCoordinate3& rhs) const
    {
        return (
                _data[0] * rhs._data[0] +
                _data[1] * rhs._data[1] +
                _data[2] * rhs._data[2]
                ) / (_data[3] * rhs._data[3]);
    }

    inline const HCoordinate3 HCoordinate3::operator ^(const HCoordinate3& rhs) const
    {
        return HCoordinate3(
                _data[1] * rhs._data[2] - _data[2] * rhs._data[1],
                _data[2] * rhs._data[0] - _data[0] * rhs._data[2],
                _data[0] * rhs._data[1] - _data[1] * rhs._data[0],
                _data[3] * rhs._data[3]);
    }

    inline HCoordinate3& HCoordinate3::operator ^=(const HCoordinate3& rhs)
    {
        GLfloat x = _data[0];
        GLfloat y = _data[1];
        GLfloat z = _data[2];

        _data[0] = y * rhs._data[2] - z * rhs._data[1];
        _data[1] = z * rhs._data[0] - x * rhs._data[2];
        _data[2] = x * rhs._data[1] - y * rhs._data[0];

        _data[3] *= rhs._data[3];
        return *this;
    }

    inline const HCoordinate3 HCoordinate3::operator *(GLfloat rhs) const
    {
        return HCoordinate3(_data[0] * rhs, _data[1] * rhs, _data[2] * rhs, _data[3]);
    }

    inline HCoordinate3& HCoordinate3::operator *=(GLfloat rhs)
    {
        _data[0] *= rhs;
        _data[1] *= rhs;
        _data[2] *= rhs;
        return *this;
    }

    inline const HCoordinate3 HCoordinate3::operator /(GLfloat rhs) const
    {
        return HCoordinate3(_data[0], _data[1], _data[2], _data[3] * rhs);
    }

    inline HCoordinate3& HCoordinate3::operator /=(GLfloat rhs)
    {
        _data[3] *= rhs;
        return *this;
    }

    // length of vector represented by this homogeneous coordinate
    inline GLfloat HCoordinate3::length() const
    {
        return std::sqrt((*this) * (*this));
    }

    inline HCoordinate3& HCoordinate3::normalize()
    {
        _data[3] *= length();
        return *this;
    }

    inline HCoordinate3::operator DCoordinate3() const
    {
        return DCoordinate3(_data[0] / _data[3], _data[1] / _data[3], _data[2] / _data[3]);
    }

    // scale from left with a scalar
    inline const HCoordinate3 operator *(GLfloat lhs, const HCoordinate3& rhs)
    {
        return rhs*lhs;
    }

    inline std::ostream& operator <<(std::ostream& lhs, const HCoordinate3& rhs)
    {
        return lhs << rhs[0] << ' ' << rhs[1] << ' ' << rhs[2] << ' ' << rhs[3];
    }

    inline std::istream& operator >>(std::istream& lhs, HCoordinate3& rhs)
    {
        return lhs >> rhs[0] >> rhs[1] >> rhs[2] >> rhs[3];
    }
}
